<?php
/**
 * @link http://www.yiiframework.com/
 * @copyright Copyright (c) 2008 Yii Software LLC
 * @license http://www.yiiframework.com/license/
 */

 ytg_Core::load('Framework_Component');

class ytg_Framework_Validator extends ytg_Framework_Component
{
    /**
     * @var array list of built-in validators (name => class or configuration)
     */
    public static $builtInValidators = array(
        'array' => 'Framework_Validator_Array',
        'boolean' => 'Framework_Validator_Boolean',
        'compare' => 'Framework_Validator_Compare',
        'date' => 'Framework_Validator_Date',
        'default' => 'Framework_Validator_DefaultValue',
        'double' => 'Framework_Validator_Number',
        'email' => 'Framework_Validator_Email',
        'exist' => 'Framework_Validator_Exist',
        'file' => 'Framework_Validator_File',
        'filter' => 'Framework_Validator_Filter',
        'image' => 'Framework_Validator_Image',
        'in' => 'Framework_Validator_Range',
        'inline' => 'Framework_Validator_Inline',
        'integer' => array(
            'class' => 'Framework_Validator_Number',
            'integerOnly' => true,
        ),
        'match' => 'Framework_Validator_RegularExpression',
        'number' => 'Framework_Validator_Number',
        'required' => 'Framework_Validator_Required',
        'safe' => 'Framework_Validator_Safe',
        'string' => 'Framework_Validator_String',
        'trim' => array(
            'class' => 'Framework_Validator_Filter',
            'filter' => 'trim',
            'skipOnArray' => true,
        ),
        'unique' => 'Framework_Validator_Unique',
        'url' => 'Framework_Validator_Url',
    );
    /**
     * @var array|string attributes to be validated by this validator. For multiple attributes,
     * please specify them as an array; for single attribute, you may use either a string or an array.
     */
    public $attributes = array();
    /**
     * @var string the user-defined error message. It may contain the following placeholders which
     * will be replaced accordingly by the validator:
     *
     * - `{attribute}`: the label of the attribute being validated
     * - `{value}`: the value of the attribute being validated
     *
     * Note that some validators may introduce other properties for error messages used when specific
     * validation conditions are not met. Please refer to individual class API documentation for details
     * about these properties. By convention, this property represents the primary error message
     * used when the most important validation condition is not met.
     */
    public $message;
    /**
     * @var array|string scenarios that the validator can be applied to. For multiple scenarios,
     * please specify them as an array; for single scenario, you may use either a string or an array.
     */
    public $on = array();
    /**
     * @var array|string scenarios that the validator should not be applied to. For multiple scenarios,
     * please specify them as an array; for single scenario, you may use either a string or an array.
     */
    public $except = array();
    /**
     * @var boolean whether this validation rule should be skipped if the attribute being validated
     * already has some validation error according to some previous rules. Defaults to true.
     */
    public $skipOnError = true;
    /**
     * @var boolean whether this validation rule should be skipped if the attribute value
     * is null or an empty string.
     */
    public $skipOnEmpty = true;
    /**
     * @var boolean whether to enable client-side validation for this validator.
     * The actual client-side validation is done via the JavaScript code returned
     * by [[clientValidateAttribute()]]. If that method returns null, even if this property
     * is true, no client-side validation will be done by this validator.
     */

    public $isEmpty;
    /**
     * @var callable a PHP callable whose return value determines whether this validator should be applied.
     * The signature of the callable should be `function ($model, $attribute)`, where `$model` and `$attribute`
     * refer to the model and the attribute currently being validated. The callable should return a boolean value.
     *
     * This property is mainly provided to support conditional validation on the server side.
     * If this property is not set, this validator will be always applied on the server side.
     *
     * The following example will enable the validator only when the country currently selected is USA:
     *
     * ```php
     * function ($model) {
     *     return $model->country == Country::USA;
     * }
     * ```
     *
     * @see whenClient
     */
    public $when;
    
    public static function createValidator($type, 
                                           ytg_Framework_Model $model, 
                                           $attributes, 
                                           $params = array())
    {
        $params['attributes'] = $attributes;

        if ($type instanceof Closure || $model->hasMethod($type)) {
            // method-based validator
            $params['class'] = 'Framework_Validator_Inline';
            $params['method'] = $type;
        } else {
            if (isset(self::$builtInValidators[$type])) {
                $type = self::$builtInValidators[$type];
            }
            if (is_array($type)) {
                $params = array_merge($type, $params);
            } else {
                $params['class'] = $type;
            }
        }

        return ytg_Core::create($params);
    }

    /**
     * @inheritdoc
     */
    public function init()
    {
        parent::init();
        $this->attributes = (array) $this->attributes;
        $this->on = (array) $this->on;
        $this->except = (array) $this->except;
    }
    
    public function validateAttributes($model, $attributes = null)
    {
        if (is_array($attributes)) {
            $attributes = array_intersect($this->attributes, $attributes);
        } else {
            $attributes = $this->attributes;
        }
        foreach ($attributes as $attribute) {
            $skip = $this->skipOnError && $model->hasErrors($attribute)
                || $this->skipOnEmpty && $this->isEmpty($model->$attribute);
            if (!$skip) {
                if ($this->when === null || call_user_func($this->when, $model, $attribute)) {
                    $this->validateAttribute($model, $attribute);
                }
            }
        }
    }
    
    public function validateAttribute($model, $attribute)
    {
        $result = $this->validateValue($model->$attribute);
        if (!empty($result)) {
            $this->addError($model, $attribute, $result[0], $result[1]);
        }
    }
    
    public function validate($value, &$error = null)
    {
        $result = $this->validateValue($value);
        if (empty($result)) {
            return true;
        }

        list($message, $params) = $result;
        $params['attribute'] = 'the input value';
        $params['value'] = is_array($value) ? 'array()' : $value;
        $error = ytg_Core::format($message, $params);

        return false;
    }

    protected function validateValue($value)
    {
        throw new Exception(get_class($this) . ' does not support validateValue().');
    }
    
    public function isActive($scenario)
    {
        return !in_array($scenario, $this->except, true) && (empty($this->on) || in_array($scenario, $this->on, true));
    }
    
    public function addError($model, $attribute, $message, $params = array())
    {
        $value = $model->$attribute;
        $params['attribute'] = $model->getAttributeLabel($attribute);
        $params['value'] = is_array($value) ? 'array()' : $value;
        $model->addError($attribute, ytg_Core::format($message, $params));
    }

    public function isEmpty($value)
    {
        if ($this->isEmpty !== null) {
            return call_user_func($this->isEmpty, $value);
        } else {
            return $value === null || $value === array() || $value === '';
        }
    }
}