<?php
ytg_Core::load('Framework_Model');
ytg_Core::load('Framework_Db_Schema');

/**
* @property string $fullTableName
*/
abstract class ytg_Framework_Db_ActiveRecord extends ytg_Framework_Model
{
    public $table;

    /**
    * Defines columns schema
    *
    * @var array
    */
    public $columns = array();

    /**
     * @var array attribute values indexed by attribute names
     */
    private $_attributes = array();
    /**
     * @var array|null old attribute values indexed by attribute names.
     * This is `null` if the record [[isNewRecord|is new]].
     */
    private $_oldAttributes;

    abstract public function schema();

    public function __construct($config=array())
    {
        if (!ytg_Framework_Db_Schema::hasTableSchema($this->table)) {
            ytg_Framework_Db_Schema::loadTableSchema($this->table,
                $this->schema());
        }

        parent::__construct($config);
    }

    public function init()
    {
        parent::init();

        $this->loadDefaultValues();
    }

    public function getFullTableName()
    {
        return $this->getTableSchema()->fullName;
    }

    public function loadDefaultValues($skipIfSet = true)
    {
        foreach ($this->getTableSchema()->columns as $column) {
            if ($column->defaultValue !== null && (!$skipIfSet || $this->{$column->name} === null)) {
                $this->{$column->name} = $column->defaultValue;
            }
        }
        return $this;
    }

    public function getTableSchema()
    {
        return ytg_Framework_Db_Schema::getTableSchema($this->table);
    }

    public function primaryKey()
    {
        return $this->getTableSchema()->primaryKey;
    }

    public function attributes()
    {
        return $this->getTableSchema()->getColumnNames();
    }

    public function __get($name)
    {
        if (isset($this->_attributes[$name]) || array_key_exists($name, $this->_attributes)) {
            return $this->_attributes[$name];
        } elseif ($this->hasAttribute($name)) {
            return null;
        }

        return parent::__get($name);
    }

    public function __set($name, $value)
    {
        if ($this->hasAttribute($name)) {
            $this->_attributes[$name] = $value;
        } else {
            parent::__set($name, $value);
        }
    }

    public function __isset($name)
    {
        try {
            return $this->__get($name) !== null;
        } catch (Exception $e) {
            return false;
        }
    }

    public function __unset($name)
    {
        if ($this->hasAttribute($name)) {
            unset($this->_attributes[$name]);
        }
    }

    public function hasAttribute($name)
    {
        return isset($this->_attributes[$name]) || in_array($name, $this->attributes());
    }

    public function getAttribute($name)
    {
        return isset($this->_attributes[$name]) ? $this->_attributes[$name] : null;
    }

    public function setAttribute($name, $value)
    {
        if ($this->hasAttribute($name)) {
            $this->_attributes[$name] = $value;
        } else {
            throw new Exception(get_class($this) . ' has no attribute named "' . $name . '".');
        }
    }

    public function getOldAttributes()
    {
        return $this->_oldAttributes === null ? array() : $this->_oldAttributes;
    }

    public function setOldAttributes($values)
    {
        $this->_oldAttributes = $values;
    }

    public function getOldAttribute($name)
    {
        return isset($this->_oldAttributes[$name]) ? $this->_oldAttributes[$name] : null;
    }

    public function setOldAttribute($name, $value)
    {
        if (isset($this->_oldAttributes[$name]) || $this->hasAttribute($name)) {
            $this->_oldAttributes[$name] = $value;
        } else {
            throw new Exception(get_class($this) . ' has no attribute named "' . $name . '".');
        }
    }

    public function markAttributeDirty($name)
    {
        unset($this->_oldAttributes[$name]);
    }

    public function isAttributeChanged($name)
    {
        if (isset($this->_attributes[$name], $this->_oldAttributes[$name])) {
            return $this->_attributes[$name] !== $this->_oldAttributes[$name];
        } else {
            return isset($this->_attributes[$name]) || isset($this->_oldAttributes[$name]);
        }
    }

    public function getDirtyAttributes($names = null)
    {
        if ($names === null) {
            $names = $this->attributes();
        }
        $names = array_flip($names);
        $attributes = array();
        if ($this->_oldAttributes === null) {
            foreach ($this->_attributes as $name => $value) {
                if (isset($names[$name])) {
                    $attributes[$name] = $value;
                }
            }
        } else {
            foreach ($this->_attributes as $name => $value) {
                if (isset($names[$name]) && (!array_key_exists($name, $this->_oldAttributes) || $value !== $this->_oldAttributes[$name])) {
                    $attributes[$name] = $value;
                }
            }
        }
        return $attributes;
    }

    public function populate($row)
    {
        $columns = $this->getTableSchema()->columns;
        foreach ($row as $name => $value) {
            if (isset($columns[$name])) {
                $this->_attributes[$name] = $columns[$name]->phpTypecast($value);
            } elseif ($this->canSetProperty($name)) {
                $this->$name = $value;
            }
        }
        $this->_oldAttributes = $this->_attributes;

        return $this;
    }

    public function findOne($condition, array $config=array())
    {
        $config['limit'] = 1;

        $result = $this->findByCondition($condition, $config);

        return $result? $result[0] : NULL;
    }

    public function findAll($condition = '', array $config=array())
    {
        return is_array($condition)
            ? $this->findByCondition($condition, $config)
            : $this->findByWhereCondition($condition, $config);
    }

    public function findByCondition($condition, array $config=array())
    {
        if (!is_array($condition)) {
            // Scalar primary key
            $primaryKey = $this->primaryKey();
            $condition = array($primaryKey[0] => $condition);
        } else if (is_integer(key($condition))) {
            // Query by primary key array
            $primaryKey = $this->primaryKey();
            $condition = array_combine($primaryKey, $condition);
        }

        $condition = $this->composeWhereCondition($condition);

        return $this->findByWhereCondition($condition, $config);
    }

    public function findByWhereCondition($condition, array $config=array())
    {
        global $wpdb;

        if (isset($config['columns'])) {
            $columns = $config['columns'];
            if (is_array($columns)) {
                foreach ($columns as $key => &$value) {
                    if (!is_int($key)) {
                        $value = "{$value} AS `{$key}`";
                    }
                }

                $columns = implode(', ', $columns);
            }
        } else {
            $columns = '*';
        }

        $alias = isset($config['alias'])? $config['alias'] : $this->table;

        $sql = "SELECT {$columns} FROM `{$this->getFullTableName()}` AS `{$alias}`";

        if (isset($config['join'])) {
            $join = $config['join'];
            if (is_array($join)) {
                $join = implode(' ', $join);
            }

            $sql .= ' ' . $join;
        }

        if ('' != $condition) {
            $sql .= " WHERE {$condition}";
        }

        if (isset($config['order'])) {
            $order = $config['order'];
            if (is_array($order)) {
                foreach ($order as $key => &$value) {
                    if (!is_int($key)) {
                        $value = $key . ' ' . $value;
                    }
                }
                $order = implode(', ', $order);
            }

            $sql .= " ORDER BY {$order}";
        }

        if (isset($config['limit'])) {
            $limit = $config['limit'];
            if (is_array($limit)) {
                $limit = implode(',', $limit);
            }

            $sql .= " LIMIT {$limit}";
        }

        $rows = $wpdb->get_results($sql, ARRAY_A);

        $indexBy = isset($config['indexBy'])
            ? $config['indexBy']
            : NULL;

        $result = array();
        foreach ($rows as $row) {
            $record = $this->createNew();
            $record->populate($row);

            $record->afterFind();

            if (!is_null($indexBy)) {
                $result[$record->$indexBy] = $record;
            } else {
                $result[] = $record;
            }
        }

        return $result;
    }

    public function count($condition=NULL)
    {
        return $this->countByWhereCondition(
            $this->composeWhereCondition($condition));
    }

    public function countByWhereCondition($condition)
    {
        global $wpdb;

        $sql = "SELECT COUNT(*) FROM `{$this->getFullTableName()}`";
        if ('' != $condition) {
            $sql .= " WHERE {$condition}";
        }

        return $wpdb->get_var($sql);
    }

    public function createNew($config=array())
    {
        $class = get_class($this);
        return new $class($config);
    }

    public function getIsNewRecord()
    {
        return $this->_oldAttributes === null;
    }

    public function setIsNewRecord($value)
    {
        $this->_oldAttributes = $value ? null : $this->_attributes;
    }

    public function save($runValidation = true, $attributeNames = null)
    {
        if ($this->getIsNewRecord()) {
            return $this->insert($runValidation, $attributeNames);
        } else {
            return $this->update($runValidation, $attributeNames) !== false;
        }
    }

    public function insert($runValidation = true, $attributes = null)
    {
        if ($runValidation && !$this->validate($attributes)) {
            return false;
        }

        if (!$this->beforeSave(true)) {
            return false;
        }

        $values = $this->getDirtyAttributes($attributes);
        if (empty($values)) {
            foreach ($this->getPrimaryKey(true) as $key => $value) {
                $values[$key] = $value;
            }
        }

        $id = $this->insertInternal($values);

        $table = $this->getTableSchema();
        if ($table->sequenceName !== null) {
            $name = $table->sequenceName;
            $id = $table->columns[$name]->phpTypecast($id);
            $this->setAttribute($name, $id);
            $values[$name] = $id;
        }

        $changedAttributes = array_fill_keys(array_keys($values), null);
        $this->setOldAttributes($values);
        $this->afterSave(true, $changedAttributes);

        return true;
    }

    public function insertInternal($attributes)
    {
        global $wpdb;

        if (!$wpdb->insert($this->getFullTableName(), $attributes)) {
            throw new Exception("Cannot insert a record into '{$this->getFullTableName()}' table: {$wpdb->last_error}");
        }

        return $wpdb->insert_id;
    }

    public function update($runValidation = true, $attributes = null)
    {
        if ($runValidation && !$this->validate($attributes)) {
            return false;
        }

        if (!$this->beforeSave(false)) {
            return false;
        }
        $values = $this->getDirtyAttributes($attributes);
        if (empty($values)) {
            $this->afterSave(false, $values);
            return 0;
        }
        $condition = $this->getOldPrimaryKey(true);

        $result = $this->updateAll($values, $condition);

        $changedAttributes = array();
        foreach ($values as $name => $value) {
            $changedAttributes[$name] = isset($this->_oldAttributes[$name]) ? $this->_oldAttributes[$name] : null;
            $this->_oldAttributes[$name] = $value;
        }
        $this->afterSave(false, $changedAttributes);

        return $result;
    }

    public function updateAll($attributes, $condition = '')
    {
        global $wpdb;

        return $wpdb->update($this->getFullTableName(), $attributes, $condition);
    }

    public function delete()
    {
        if (!$this->beforeDelete()) {
            return false;
        }

        // we do not check the return value of deleteAll() because it's possible
        // the record is already deleted in the database and thus the method will return 0
        $condition = $this->getOldPrimaryKey(true);

        $this->deleteAll($condition);

        $this->setOldAttributes(null);
        $this->afterDelete();

        return $this;
    }

    public function deleteAll($condition)
    {
        return $this->deleteByWhereCondition(
            $this->composeWhereCondition($condition));
    }

    public function deleteByWhereCondition($condition)
    {
        global $wpdb;

        $sql = "DELETE FROM `{$this->getFullTableName()}` WHERE {$condition}";
        return $wpdb->query($sql);
    }

    public function equals($record)
    {
        if ($this->getIsNewRecord() || $record->getIsNewRecord()) {
            return false;
        }

        return get_class($this) === get_class($record) && $this->getPrimaryKey() === $record->getPrimaryKey();
    }

    public function getPrimaryKey($asArray = false)
    {
        $keys = $this->primaryKey();
        if (!$asArray && count($keys) === 1) {
            return isset($this->_attributes[$keys[0]]) ? $this->_attributes[$keys[0]] : null;
        } else {
            $values = array();
            foreach ($keys as $name) {
                $values[$name] = isset($this->_attributes[$name]) ? $this->_attributes[$name] : null;
            }

            return $values;
        }
    }

    public function getOldPrimaryKey($asArray = false)
    {
        $keys = $this->primaryKey();
        if (empty($keys)) {
            throw new Exception(get_class($this) . ' does not have a primary key. You should either define a primary key for the corresponding table or override the primaryKey() method.');
        }
        if (!$asArray && count($keys) === 1) {
            return isset($this->_oldAttributes[$keys[0]]) ? $this->_oldAttributes[$keys[0]] : null;
        } else {
            $values = array();
            foreach ($keys as $name) {
                $values[$name] = isset($this->_oldAttributes[$name]) ? $this->_oldAttributes[$name] : null;
            }

            return $values;
        }
    }
    
    public function refresh()
    {
        /* @var $record BaseActiveRecord */
        $record = $this->findOne($this->getPrimaryKey(true));
        if ($record === null) {
            return false;
        }
        foreach ($this->attributes() as $name) {
            $this->_attributes[$name] = isset($record->_attributes[$name]) ? $record->_attributes[$name] : null;
        }
        $this->_oldAttributes = $this->_attributes;

        return true;
    }

    //
    // Events
    //

    public function afterFind()
    {

    }

    public function beforeSave($insert)
    {
        return TRUE;
    }

    public function afterSave($insert, $changedAttributes)
    {

    }

    public function beforeDelete()
    {
        return TRUE;
    }

    public function afterDelete()
    {

    }

    //
    // Relations
    //

    public function hasOne($class, $link, array $config=array())
    {
        $filter = array();
        foreach ($link as $external => $internal) {
            $value = $this->getAttribute($internal);
            if (!is_null($value)) {
                $filter[$external] = $value;
            }
        }

        if (!$filter) {
            return NULL;
        }

        return ytg_Core::model($class)->findOne($filter, $config);
    }

    public function hasMany($class, $link, array $config=array())
    {
        $filter = array();
        foreach ($link as $external => $internal) {
            $value = $this->getAttribute($internal);
            if (!is_null($value)) {
                $filter[$external] = $value;
            }
        }

        if (!$filter) {
            return array();
        }

        return ytg_Core::model($class)->findAll($filter, $config);
    }

    //
    // Helpers
    //

    public function composeWhereCondition($condition, $operator = 'AND')
    {
        global $wpdb;

        if (!$condition) {
            return NULL;
        }
        if (!is_array($condition)) {
            return $condition;
        }

        $where = array();
        $params = array();

        foreach ($condition as $name=>$value) {
            if (is_null($value)) {
                continue;
            }

            $params[] = $value;
            if (is_numeric($value)) {
                $where[] = "`{$name}` = %d";
            } else {
                $where[] = "`{$name}` = '%s'";
            }
        }
        return $wpdb->prepare(implode(" {$operator} ", $where), $params);
    }

    public function joinWhereConditions(array $conditions, $operator='AND')
    {
        $result = array();

        foreach ($conditions as $condition) {
            if (is_null($condition)) {
                continue;
            }

            $where = $this->composeWhereCondition($condition);
            if ('' == $where) {
                continue;
            }

            $result[] = "({$where})";
        }

        return implode(" {$operator} ", $result);
    }
}
