<?php

/**
 * Class ytg_Framework_Core
 * @property string $baseName;
 * @property string $basePath;
 * @property string $baseUrl;
 *
 * Components:
 * @property ytg_Framework_Component_Cache $cache
 * @property ytg_Framework_Component_Meta $meta
 * @property ytg_Framework_Component_Options $options
 * @property ytg_Framework_Component_PostTypes $postTypes
 * @property ytg_Framework_Component_UserSession $userSession
 * @property ytg_Framework_Component_View $view
 */
class ytg_Framework_Core
{
    /**
    * @var ytg_Framework_Core
    */
    static public $app;

    public $path;
    public $slug;
    public $prefix;
    public $version;

    public $config = array();
    public $registry = array();

    /**
    * @property string $path Main plugin file path.
    * @property string $baseName plugin_basename result for the plugin.
    * @property string $basePath Plugin directory path.
    * @property string $baseUrl Plugin directory URL.
    */
    protected $_baseName;
    protected $_basePath;
    protected $_baseUrl;

    public $contentCache = array();

    protected $_namespaces = array(
        'Framework' => 'library/Framework',
    );

    public function __construct()
    {
        if (!self::$app) {
            self::$app = $this;
        }

        // Init config
        $this->_configure();

        // Init components
        $this->_bootstrap();
    }

    //
    // Paths
    //

    public function getBaseName()
    {
        if (is_null($this->_baseName)) {
            $this->_baseName = plugin_basename($this->path);
        }

        return $this->_baseName;
    }

    public function getBasePath()
    {
        if (is_null($this->_basePath)) {
            $this->_basePath = dirname($this->path);
        }

        return $this->_basePath;
    }

    public function getBaseUrl()
    {
        if (is_null($this->_baseUrl)) {
            $this->_baseUrl = plugin_dir_url($this->path);
        }

        return $this->_baseUrl;
    }

    static public function getPluginTitle()
    {
        return self::config('plugin/title');
    }

    //
    // __get and __set
    //

    public function __get($name)
    {
        $method = 'get' . ucfirst($name);
        if (method_exists($this, $method)) {
            return $this->$method();
        }

        return $this->_loadComponent($name);
    }

    public function __set($name, $value)
    {
        $method = 'set' . ucfirst($name);
        if (method_exists($this, $method)) {
            $this->$method($value);
        } else {
            $this->$name = $value;
        }

        return $this;
    }

    //
    // Components
    //

    protected function _bootstrap()
    {
        foreach ($this->getConfigValue('bootstrap', array()) as $name) {
            $this->_loadComponent($name);
            if ($this->getRegistryValue('abort')) {
                $this->unsetRegistryValue('abort');
                break;
            }
        }
    }

    protected function _loadComponent($name)
    {
        $components = $this->getConfigValue('components', array());
        $component = isset($components[$name])
            ? $components[$name]
            : array();

        if (!isset($component['class'])) {
            $component['class'] = ytg_Core::fallback(
                'Component_' . ucfirst($name));
        }

        $result = $this->createObject($component);
        $this->$name = $result;

        return $result;
    }

    //
    // Config
    //

    protected function _configure()
    {
        $this->config = $this->loadConfig('main');
    }

    public function setConfigValue($path, $value)
    {
        $this->config[$path] = $value;

        return $this;
    }

    public function getConfigValue($path, $default=NULL)
    {
        return isset($this->config[$path])
            ? $this->config[$path]
            : $default;
    }

    /**
    * Static alias for getConfigValue
    */
    static public function config($path, $default=NULL)
    {
        return self::$app->getConfigValue($path, $default);
    }

    public function getConfigValues($searchPath)
    {
        if ('/' != substr($searchPath, -1)) {
            $searchPath .= '/';
        }

        $depth = substr_count($searchPath, '/');
        $length = strlen($searchPath);

        $result = array();
        foreach ($this->config as $path=> $value) {
            if (0 === strpos($path, $searchPath)
                && $depth == substr_count($path, '/')
            ) {
                $result[substr($path, $length)] = $value;
            }
        }

        return $result;
    }

    /**
    * Static alias for getConfigValues
    */
    static public function configSection($path)
    {
        return self::$app->getConfigValues($path);
    }

    public function loadConfig($path)
    {
        $result = require "{$this->basePath}/app/config/{$path}.php";

        // Look for custom config
        $custom = WP_CONTENT_DIR . '/custom/' . $this->slug
            . "/config/{$path}.php";
        if (is_file($custom)) {
            $custom = require $custom;
            $result = array_merge($result, $custom);
        }

        return $result;
    }

    /**
    * Registry
    */

    public function getRegistryValue($key, $default=NULL)
    {
        return isset($this->registry[$key])
            ? $this->registry[$key]
            : $default;
    }

    public function setRegistryValue($key, $value)
    {
        $this->registry[$key] = $value;

        return $this;
    }

    public function unsetRegistryValue($path)
    {
        unset($this->registry[$path]);
    }

    //
    // Loader
    //

    /**
    * Static version of importClass
    *
    * @param string $class Class name
    */
    static public function load($class)
    {
        return self::$app->loadClass($class);
    }

    public function loadClass($class)
    {
        $file = $this->findClass($class);
        if (!file_exists($file)) {
            throw new Exception("Class file '{$file}' not found.");
        }
        require_once $file;

        // Prefix class name if not
        if ($this->prefix . '_' != substr($class, 0, strlen($this->prefix)+1)) {
            $class = $this->prefix . '_' . $class;
        }

        return $class;
    }

    public function findClass($class)
    {
        $path = explode('_', $class);

        // Remove prefix if prefixed
        if ($this->prefix == $path[0]) {
            array_shift($path);
        }

        if (isset($this->_namespaces[$path[0]])) {
            $subPath = $this->_namespaces[$path[0]];
            unset($path[0]);
        } else {
            $subPath = 'app/classes';
        }

        return "{$this->basePath}/{$subPath}/" . implode('/', $path) . '.php';
    }

    static public function fallback($class, $fallback = NULL)
    {
        return self::$app->findClassFallback($class, $fallback);
    }

    public function findClassFallback($class, $fallback = NULL)
    {
        $file = $this->findClass($class);
        if (is_file($file)) {
            return $class;
        }

        if (is_null($fallback)) {
            $fallback = array('Framework_');
        }

        foreach ($fallback as $prefix) {
            $result = $prefix . $class;
            $file = $this->findClass($result);
            if (is_file($file)) {
                return $result;
            }
        }

        throw new Exception("Class '{$class}' not found.");
    }

    /**
    * Static wrapper for createObject
    *
    * @param string $class Class name
    */
    static public function create($class, $config=array())
    {
        return self::$app->createObject($class, $config);
    }

    public function createObject($class, $config=array())
    {
        if (is_array($class)) {
            $config = $class;
            $class = $config['class'];
            unset($config['class']);

            return $this->createObject($class, $config);
        }

        $class = $this->loadClass($class);

        return new $class($config);
    }

    static public function model($class)
    {
        $class = self::$app->loadClass($class);

        return call_user_func(array($class, 'model'), $class);
    }

    static public function call($callable, array $params = array())
    {
        // Load class for static method filters
        if (is_array($callable)
            && isset($callable[0])
            && is_string($callable[0])
            && !class_exists($callable[0])
        ) {
            $callable[0] = self::load($callable[0]);
        }

        return call_user_func_array($callable, $params);
    }

    static public function mapMethodParams($class, $method, array $params)
    {
        if (!method_exists($class, $method)) {
            throw new Exception("Method '{$method}' not found.");
        }

        $result = array();
        $method = new ReflectionMethod($class, $method);

        foreach($method->getParameters() as $param) {
            $name = $param->getName();
            if (array_key_exists($name, $params)) {
                if ($param->isArray() && !is_array($params[$name])) {
                    throw new Exception("Parameter '{$name}' is invalid.");
                }
                $result[] = $params[$name];
            } else if($param->isDefaultValueAvailable()) {
                $result[] = $param->getDefaultValue();
            } else {
                throw new Exception("Parameter '{$name}' is required.");
            }
        }

        return $result;
    }

    //
    // Dispatcher
    //

    public function __call($method, $params)
    {
        $parts = $this->_parseMethodCall($method);

        switch ($parts[0]) {
            case 'dispatch':
                return $this->dispatch($parts[1], $parts[2], $params);
        }

        throw new Exception("Method {$method} not found.");
    }

    public function dispatch($controller, $action=NULL, $params=array())
    {
        $controller = (array) $controller;
        $className = array_map('ucfirst', $controller);
        $className = implode('_', $className);

        /**
        * @var ytg_Framework_Controller
        */
        $controllerObj = $this->createObject("Controller_{$className}");

        return $controllerObj->dispatch(implode('/', $controller), $action,
            $params);
    }

    protected function _parseMethodCall($name)
    {
        $parts = explode('_', $name);

        if (count($parts) < 2) {
            return NULL;
        }

        $result[0] = array_shift($parts);
        $last = count($parts) > 1? array_pop($parts) : NULL;

        $result[1] = $parts;
        $result[2] = $last;

        return $result;
    }

    /**
    * Helpers
    */

    static public function format($text, array $values)
    {
        foreach ($values as $name=>$value) {
            $text = str_replace('{' . $name . '}', $value, $text);
        }

        return $text;
    }

    /**
    * Returns true if processing AJAX request
    */
    static public function isAjaxRequest()
    {
        return defined('DOING_AJAX') && DOING_AJAX;
    }

    static public function terminate($message, $statusCode)
    {
        wp_die($message, $statusCode);
    }

    /**
    * Plugin singleton
    */

    static public function init()
    {
        return new ytg_Core;
    }
}