<?php
ytg_Core::load('Framework_Component');

class ytg_Component_Process extends ytg_Framework_Component
{
    public $sessionNamespace;

    public $worker;

    public $maxLogLines = 1000; //Max lines to keep in memory

    protected $_lineCount = 0;

    public function init()
    {
        parent::init();

        $activate = ytg_Core::$app->activate;
        if (!$activate->activated 
            || '' == $activate->email
            || $activate->email != ytg_Core::$app->options->get("activate_email-{$activate->productName}")
        ) {
            return;
        }

        if (is_null($this->sessionNamespace)) {
            $this->sessionNamespace = ytg_Core::$app->prefix . '_process';
        }
    }

    public function getIsActive()
    {
        return isset($_SESSION[$this->sessionNamespace])
            && (!isset($_SESSION[$this->sessionNamespace]['command'])
                || 'terminate' != $_SESSION[$this->sessionNamespace]['command']);
    }

    public function run(ytg_Service_Worker $worker, $sessionNamespace=NULL)
    {
        $this->worker = $worker;

        $worker->process = $this;

        $oldSessionNamespace = NULL;
        if (!is_null($sessionNamespace)) {
            $oldSessionNamespace = $this->sessionNamespace;
            $this->sessionNamespace = $sessionNamespace;
        }

        error_reporting(E_ERROR|E_PARSE|E_CORE_ERROR|E_COMPILE_ERROR|E_USER_ERROR|E_RECOVERABLE_ERROR);
        ini_set('error_log', ytg_Core::$app->basePath . '/app/runtime/error.txt');
        ini_set('log_errors', TRUE);
        ytg_Core::$app->errorHandler->enable(E_ERROR|E_WARNING);

        $lock = $this->_openLockFile(get_class($worker) . '.lock');

        @set_time_limit(36000);
        @ini_set('memory_limit', '512M');

        try {
            $this->startOutput();
            $this->feedback(array(
                'status' => 'progress'
            ));
            $this->ping();

            $worker->run();

            $this->feedback(array(
                'log' => "\nDone.",
                'status' => 'success',
                'statusText' => 'Done.',
            ));

        } catch (ytg_Component_Process_FailedException $e) {
            $this->feedback(array(
                'log' => "{$e->getMessage()}\n\nFailed.",
                'status' => 'error',
                'statusText' => "Failed. {$e->getMessage()}.",
            ));

        } catch (ytg_Component_Process_FatalException $e) {
            $this->feedback(array(
                'log' => "{$e->getMessage()}\n\nFailed.",
                'status' => 'error',
                'statusText' => "Error. {$e->getMessage()}.",
            ));

        } catch (ytg_Component_Process_CriticalException $e) {
            $this->feedback("Critical error: {$e->getMessage()}\n\n");
            $this->feedback("Debug trace: \n{$e->getTraceAsString()}\n\n");
            $this->feedback(array(
                'log' => "Critical error occurred. Please contact plugin support.",
                'status' => 'error',
                'statusText' => "Critical error occurred. Please contact plugin support.",
            ));

            header('HTTP/1.1 500 Internal Server Error', TRUE, 500);

            @error_log($e, E_ERROR);
        } catch (Exception $e) {
            $this->feedback("Unexpected error: {$e}\n\n");
            $this->feedback(array(
                'log' => "Unexpected error occurred. Please contact plugin support.",
                'status' => 'error',
                'statusText' => "Unexpected error occurred. Please contact plugin support.",
            ));

            header('HTTP/1.1 500 Internal Server Error', TRUE, 500);

            @error_log($e, E_ERROR);
        }

        ytg_Core::$app->errorHandler->disable();

        $this->_closeLockFile($lock);

        if (!is_null($oldSessionNamespace)) {
            $this->sessionNamespace = $oldSessionNamespace;
        }

        $this->worker = NULL;
    }

    protected function _openLockFile($name)
    {
        $path = ytg_Core::$app->basePath . '/app/runtime/' . $name;
        $handle = $this->_openLockFileInternal($path);
        if (!$handle) {
            $uploadDir = wp_upload_dir();
            $path = $uploadDir['basedir'] . '/' . $name;
            $handle = $this->_openLockFileInternal($path);
        }

        if (!flock($handle, LOCK_EX|LOCK_NB)) {
            exit('Another instance in running');
        }

        return compact('handle', 'path');
    }

    protected function _openLockFileInternal($path)
    {
        $dir = dirname($path);

        if (!is_dir($dir)) {
            @mkdir($dir, 0777, TRUE);
        } else {
            @chmod($dir, 0777);
        }

        return @fopen($path, 'w');
    }

    protected function _closeLockFile(array $lock)
    {
        if ($lock['handle']) {
            @fclose($lock['handle']);
        }
        @unlink($lock['path']);
    }

    public function feedback($options = array())
    {
        $command = $this->_feedbackExchange($options);

        if ('terminate' == $command) {
            $this->_feedbackTerminate();
            exit;
        }
    }

    public function startOutput()
    {
        @session_start();

        $_SESSION[$this->sessionNamespace] = array(
            'log' => '',
            'logStart' => 0,
            'status' => NULL,
            'command' => NULL,
        );

        $this->_lineCount = 0;

        return $this;
    }

    protected function _feedbackExchange($options = array())
    {
        @session_start();

        if (!isset($_SESSION[$this->sessionNamespace])) {
            session_write_close();
            return 'terminate';
        }

        $result = isset($_SESSION[$this->sessionNamespace]['command'])
            ? $_SESSION[$this->sessionNamespace]['command']
            : NULL;

        if (!is_array($options)) {
            $options = array('log' => $options);
        }

        if (isset($options['log'])) {
            // Count lines
            $this->_lineCount += substr_count($options['log'], "\n");
            $_SESSION[$this->sessionNamespace]['log'] .= $options['log'];
            // Truncate log if needed
            if ($this->_lineCount > $this->maxLogLines) {
                $this->_truncateLog();
            }
        }

        if (isset($options['status'])) {
            $_SESSION[$this->sessionNamespace]['status'] = $options['status'];
        }

        if (isset($options['statusText'])) {
            $_SESSION[$this->sessionNamespace]['statusText'] = $options['statusText'];
        }

        session_write_close();

        return $result;
    }

    protected function _truncateLog()
    {
        $newLineCount = floor($this->maxLogLines / 2);
        $number = $this->_lineCount - $newLineCount;
        $this->_lineCount = $newLineCount;
        $offset = $this->_strposX($_SESSION[$this->sessionNamespace]['log'],
            "\n", $number);
        if (FALSE == $offset) {
            throw new Exception('Error truncating log');
        }

        $_SESSION[$this->sessionNamespace]['logStart'] += $offset + 1;
        $_SESSION[$this->sessionNamespace]['log'] = substr(
            $_SESSION[$this->sessionNamespace]['log'], $offset+1);
    }

    /**
     * Find the position of the Xth occurrence of a substring in a string
     * @param $haystack
     * @param $needle
     * @param $number integer > 0
     * @return int
     */
    protected function _strposX($haystack, $needle, $number = 1)
    {
        $offset = 0;
        for ($i = 0; $i < $number; $i++) {
            $result = strpos($haystack, $needle, $offset);
            if (FALSE === $result) {
                return FALSE;
            }

            $offset = $result + strlen($needle);
        }

        return $result;
    }

    protected function _feedbackTerminate()
    {
        @session_start();

        unset($_SESSION[$this->sessionNamespace]);
        session_write_close();
    }

    /**
     * Workaround for rapidly disconnecting DB servers
     */
    public function ping()
    {
        global $wpdb;

        if (method_exists($wpdb, 'check_connection')) {
            if (!$wpdb->check_connection(FALSE)) {
                throw new Exception('Database connection is lost.');
            }
        } else {
            @mysql_ping();
        }

        return $this;
    }

    public function getOutput($logOffset = 0)
    {
        $result = isset($_SESSION[$this->sessionNamespace])
            ? $_SESSION[$this->sessionNamespace]
            : NULL;

        $logStart = $_SESSION[$this->sessionNamespace]['logStart'];

        $start = $logOffset - $logStart;
        if ($start >= 0) {
            $result['logTruncated'] = FALSE;
        } else {
            $result['logTruncated'] = TRUE;
            $start = 0;
        }

        $result['logOffset'] = $logStart + strlen($result['log']);
        $result['log'] = substr($result['log'], $start);

        return $result;
    }

    public function sendCommand($command)
    {
        if (!isset($_SESSION[$this->sessionNamespace])) {
            $_SESSION[$this->sessionNamespace] = array();
        }
        $_SESSION[$this->sessionNamespace]['command'] = $command;
    }

    public function terminate()
    {
        $this->sendCommand('terminate');
    }

    public function clearOutput()
    {
        unset($_SESSION[$this->sessionNamespace]);
    }
}

class ytg_Component_Process_Exception extends Exception
{

}

/**
 * Non-fatal error.
 * For example, connection timeout
 * The application may skip to another item.
 */
class ytg_Component_Process_ItemException
    extends ytg_Component_Process_Exception
{

}

/**
 * Non-critical error
 */
class ytg_Component_Process_FailedException
    extends ytg_Component_Process_Exception
{

}

/**
 * Sign of bug or misconfiguration.
 * For example, required option didn't set.
 * The application have to stop and ask user to contact support.
 */
class ytg_Component_Process_CriticalException
    extends ytg_Component_Process_Exception
{

}

/**
 * Fatal error.
 * For example, invalid API credentials.
 * The application cannot skip to another item and have to warn the user.
 */
class ytg_Component_Process_FatalException
    extends ytg_Component_Process_Exception
{

}