<?php
ytg_Core::load('Framework_Component');

class ytg_Component_PostTemplate extends ytg_Framework_Component
{
    public $templateMap = array(
        'video-player' => 'generateVideoPlayer',
        'description' => 'generateDescription',
    );

    public $linkPattern = '~(<a\s[^>]*href=)(\"??)([^\">]*?)(\\2[^>]*>.*<\/a>)~isuU';

    /**
     * @var WP_Post
     */
    protected $post;

    /**
     * @var string
     */
    protected $content;

    /**
     * @var ytg_Framework_Component_Meta
     */
    protected $meta;

    protected $_linkIndex;

    protected $_redirectPath;

    protected $_postMetaCache = array();

    public function init()
    {
        $activate = ytg_Core::$app->activate;
        if (!$activate->activated || '' == $activate->email
            || $activate->email != get_option("ytg_activate_email-{$activate->productName}")
        ) {
            return;
        }

        add_filter('the_post', array($this, '_filterPost'));
        add_filter('the_content_more_link', array($this, '_filterContentMoreLink'));
        add_filter('the_content', array($this, '_filterContent'), -300);

        // URL listener
        if (!is_admin() && ytg_Core::$app->options->get('displayCloakLinks')) {
            $this->_redirectPath = $this->_checkCloakUrl();
            if ($this->_redirectPath) {
                add_action('wp_loaded', array($this, '_redirectCloakUrl'));
            }
        }

        add_filter('pre_get_avatar_data', array($this, '_getAvatarData'), 10, 2);

        add_filter('get_post_metadata', array($this, '_getPostMetadata'), 10, 4);
    }

    public function _filterPost($value)
    {
        $this->post = $this->isOwnPost($value)
            ? $value
            : NULL;
    }

    public function _filterContentMoreLink($value)
    {
        $this->post = NULL;
    }

    public function _filterContent($value)
    {
        if (ytg_Core::$app->demo && !current_user_can('administrator')) {
            return $value;
        }

        if (!$this->post) {
            return $value;
        }

        $this->content = $value;
        // Workaround for themes passing empty content
        if ('' == $this->content) {
            $this->content = get_the_content();
        }

        $this->meta = ytg_Core::$app->create('Framework_Component_Meta', array(
            'id' => $this->post->ID,
        ));

        /**
         * @var ytg_Model_Template $template
         */
        $template = ytg_Core::$app->model('Model_Template')->find(
            $this->meta->get('postTemplate'));
        $value = $template->content;

        foreach ($this->templateMap as $key => $method) {
            $search = "[{$key}]";
            if (FALSE === strpos($value, $search)) {
                continue;
            }

            $value = str_replace($search, $this->$method(), $value);
        }

        // Cloak links
        if (ytg_Core::$app->options->get('displayCloakLinks')) {
            $this->_linkIndex = 0;
            $this->meta->delete('cloakedLinks');
            $value = preg_replace_callback($this->linkPattern,
                array($this, '_cloakCallback'), $value);
        }

        return $value;
    }

    public function isOwnPost($post)
    {
        return '' != get_post_meta($post->ID, 'ytg_videoId', TRUE);
    }

    public function generateVideoPlayer()
    {
        $theme = get_template();

        return $this->stripNewLines(
            ytg_Core::$app->view->render('post-template/video-player', array(
                'post' => $this->post,
                'content' => $this->content,
                'meta' => $this->meta,
                'plain' => 'vidizon-pro' == $theme,
            )));
    }

    public function generateDescription()
    {
        return $this->content;
    }

    public function _cloakCallback($match)
    {
        $this->meta->add('cloakedLinks', trim($match[3]));
        $this->_linkIndex++;

        #3 is link URL
        $match[3] = $this->_generateCloakLink(array($this->_linkIndex));

        // Remove whole match from implode
        unset($match[0]);

        return implode('', $match);
    }

    protected function _generateCloakLink(array $extraDirs = NULL)
    {
        global $wp_rewrite;

        $result = array(untrailingslashit(site_url()));

        if (!$wp_rewrite->using_mod_rewrite_permalinks()) {
            $result[] = $wp_rewrite->index;
        }

        $result[] = ytg_Core::$app->options->get('displayCloakingDirectory');
        $result[] = $this->post->post_name;

        if ($extraDirs) {
            $result = array_merge($result, $extraDirs);
        }

        $result = implode('/', $result);

        return $result;
    }

    public function _checkCloakUrl()
    {
        $result = explode('/', substr($_SERVER['REQUEST_URI'],
            strlen($this->_getHomePath())));

        if ('index.php' == $result[0]) {
            array_shift($result);
        }

        if (count($result) < 2
            || ytg_Core::$app->options->get('displayCloakingDirectory') !== $result[0]
            || '' == $result[1]
        ) {
            return NULL;
        }

        array_shift($result);

        return $result;
    }

    public function _redirectCloakUrl()
    {
        $slug = $this->_redirectPath[0];

        $results = get_posts(array(
            'name' => $slug,
            'post_status' => array('publish', 'pending', 'draft', 'future'),
            'posts_per_page' => 1,
            'orderby' => 'none',
        ));
        if (!$results) {
            return;
        }

        /**
         * @var WP_Post $post
         */
        $post = $results[0];

        if (!$this->isOwnPost($post)) {
            return;
        }

        $linkIndex = $this->_redirectPath[1] - 1;
        $links = get_post_meta($post->ID, 'ytg_cloakedLinks');
        if (!isset($links[$linkIndex])) {
            return;
        }

        echo ytg_Core::$app->view->render('post-template/redirect', array(
            'url' => $links[$linkIndex],
        ));

        exit;
    }

    protected function _getHomePath()
    {
        return trailingslashit((string)parse_url(site_url(), PHP_URL_PATH));
    }

    protected function stripNewLines($text)
    {
        return str_replace(array("\r", "\n"), ' ', $text);
    }

    //
    // Avatars
    //

    public function _getAvatarData($args, $id_or_email)
    {
        if ($id_or_email instanceof WP_Comment) {
            $meta = ytg_Core::create('Framework_Component_Meta', array(
                'type' => 'comment',
                'id' => $id_or_email->comment_ID,
            ));

            $info = $meta->get('info');
            if (!is_array($info) || !isset($info['authorProfileImageUrl'])
                || $this->_isDefaultAvatarUrl($info['authorProfileImageUrl'])
            ) {
                return $args;
            }

            $url = $info['authorProfileImageUrl'];
            if (isset($args['size'])) {
                $url = add_query_arg('sz', $args['size'], $url);
            }

            $args['url'] = $url;
        }

        return $args;
    }

    protected function _isDefaultAvatarUrl($url)
    {
        return 'https://lh3.googleusercontent.com/-XdUIqdMkCWA/AAAAAAAAAAI/AAAAAAAAAAA/4252rscbv5M/photo.jpg?sz=50' == $url;
    }

    public function _getPostMetadata($value, $objectId, $metaKey, $single)
    {
        if (!$value && 'tm_video_url' == $metaKey) {
            if (!isset($this->_postMetaCache[$objectId][$metaKey])) {
                $videoId = get_post_meta($objectId, 'ytg_videoId', TRUE);
                if ('' != $videoId) {
                    $value = "https://www.youtube.com/watch?v={$videoId}";
                }

                $this->_postMetaCache[$objectId][$metaKey] = $value;
            } else {
                $value = $this->_postMetaCache[$objectId][$metaKey];
            }

            if (!$single) {
                $value = array($value);
            }
        }

        return $value;
    }
}