<?php
ytg_Core::load('Service_Worker');
ytg_Core::load('Component_Process'); // for exception classes if run directly

/**
 * Accessible through getOption():
 * @property string $keyword
 * @property string[] $negativeKeywords
 * @property string[] $channels
 * @property string $sortOrder
 * @property string $filterDuration
 * @property integer $minViews
 * @property integer $maxCount
 * @property boolean $fetchComments
 * @property string $publicationMethod
 * @property integer $postsPerWeek
 * @property string[] $postCategories
 * @property integer $postTemplate
 * @property string $postType
 *
 * @property boolean $generatorFeaturedImages
 * @property boolean $generatorRemoveLinks
 * @property boolean $generatorFetchComments
 */
class ytg_Component_Generator extends ytg_Service_Worker
{
    protected $_forceOptions = array();

    public $options = array();

    public $allowTextSpinningOff = FALSE;

    public $videoIds;

    public $searchMaxResults = 50;

    public $previewCount = 200;

    public $log;

    protected $_enableCheckVideoInfo;

    protected $_videoInfoParts = 'snippet';

    protected $_counts;

    public function init()
    {
        parent::init();

        global $wpdb;

        // Workaround for connection timeout on HostGator
        $wpdb->query("set session wait_timeout=600");
    }

    public function run()
    {
        // Prepare search parameters
        $this->feedback(array(
            'log' => 'Starting... ',
            'statusText' => 'Starting...',
        ));

        $this->_forceOptions = array();

        /**
         * @var ytg_Component_Client_Youtube $youtube
         */
        $youtube = ytg_Core::$app->client_Youtube;

        // Prepare search parameters
        $searchParams = $this->_prepareSearchParams();
        $this->_prepareCheckVideoInfo();

        $this->_initCounts();

        $this->feedback("ok.\n\n");

        $this->feedback(array(
            'statusText' => 'Generating posts...',
        ));

        if (!$this->videoIds) {
            foreach (explode("\n", $this->channels) as $channel) {
                if ('' != $channel) {
                    $searchParams['channelId'] = $channel;
                    $this->feedback("Searching channel {$channel}:\n\n");
                } else {
                    unset($searchParams['channelId']);
                }

                unset ($searchParams['pageToken']);

                while ($this->_counts['published'] < $this->maxCount) {
                    // Search request
                    $this->feedback("Searching... ");
                    try {
                        $response = $youtube->search($searchParams);
                    } catch (ytg_Component_Client_Youtube_Exception $e) {
                        if ('invalidChannelId' == $e->reason) {
                            $this->feedback("error: Channel '{$channel}' is invalid. For more info, visit this page: http://www.georgekatsoudas.com/contact/knowledgebase.php?article=178\n\n");
                            break;
                        }

                        throw new ytg_Component_Process_FatalException(
                            "YouTube API request failed: {$e->getMessage()}");
                    }

                    if (!$response['items'] && !isset($searchParams['pageToken'])) {
                        $this->feedback("no results found.\n\n");
                        break;
                    }

                    $count = isset($response['items']) && is_array($response['items'])
                        ? count($response['items'])
                        : 0;
                    $this->feedback("found videos: {$count}.\n\n");

                    foreach ($response['items'] as $videoInfo) {
                        $this->_generatePostWithProgress($videoInfo['id']['videoId']);

                        $progress = round(100 *
                            ($this->_counts['published'] / $this->maxCount));

                        // Extra line break at the end of post generation
                        $this->feedback(array(
                            'log' => "\n",
                            'statusText' => 'Generating posts...'
                                . ($progress > 0 ? " {$progress}%" : ''),
                        ));

                        if ($this->_counts['published'] >= $this->maxCount) {
                            break;
                        }
                    }

                    if (!isset($response['nextPageToken']) || !$response['items']) {
                        unset ($searchParams['pageToken']);
                        $this->feedback("The search reached the end of results.\n\n");
                        break;
                    }

                    $searchParams['pageToken'] = $response['nextPageToken'];
                }

                if ($this->_counts['published'] >= $this->maxCount) {
                    break;
                }
            }
        } else { // Add specific video IDs
            $count = count($this->videoIds);
            foreach ($this->videoIds as $videoId) {
                $this->_generatePostWithProgress($videoId);

                $progress = round(100 *
                    ($this->_counts['processed'] /$count));

                // Extra line break at the end of post generation
                $this->feedback(array(
                    'log' => "\n",
                    'statusText' => 'Generating posts...'
                        . ($progress > 0 ? " {$progress}%" : ''),
                ));
            }
        }

        $this->feedback("Finished. Generated posts: {$this->_counts['published']}. "
            . "Processed videos: {$this->_counts['processed']}, "
            . "already published: {$this->_counts['duplicate']}, "
            . "filtered out: {$this->_counts['filtered']}, "
            . "errors: {$this->_counts['error']}.");
    }

    public function preview()
    {
        $this->_forceOptions = array();

        $youtube = ytg_Core::$app->client_Youtube;

        $searchParams = $this->_prepareSearchParams();

        $result = array();
        if (!$this->videoIds) {
            foreach (explode("\n", $this->channels) as $channel) {
                if ('' != $channel) {
                    $searchParams['channelId'] = $channel;
                } else {
                    unset($searchParams['channelId']);
                }

                unset ($searchParams['pageToken']);

                while (count($result) < $this->previewCount) {
                    try {
                        $response = $youtube->search($searchParams);
                    } catch (ytg_Component_Client_Youtube_Exception $e) {
                        throw new ytg_Component_Generator_Exception(
                            "YouTube API request failed: {$e->getMessage()}", 0, $e);
                    }

                    foreach ($response['items'] as $videoInfo) {
                        $videoId = $videoInfo['id']['videoId'];
                        $result[] = array(
                            'videoId' => $videoId,
                            'title' => $videoInfo['snippet']['title'],
                            'duplicate' => (bool) $this->findVideoPost($videoId),
                        );
                    }

                    if (isset($response['nextPageToken'])) {
                        $searchParams['pageToken'] = $response['nextPageToken'];
                    } else {
                        unset ($searchParams['pageToken']);
                        $this->feedback("The search reached the end of results.\n\n");
                        break;
                    }
                }

                if (count($result) >= $this->previewCount) {
                    break;
                }
            }
        } else { // scan videoIds
            foreach ($this->videoIds as $videoId) {
                try {
                    $videoInfoResponse = ytg_Core::$app->client_Youtube->request('videos', array(
                        'part' => $this->_videoInfoParts,
                        'id' => $videoId,
                    ));
                } catch (ytg_Component_Client_Youtube_Exception $e) {
                    throw new ytg_Component_Generator_Exception(
                        "YouTube API request failed: {$e->getMessage()}", 0, $e);
                }

                if (!isset($videoInfoResponse['items'][0])) {
                    throw new ytg_Component_Generator_Exception("Video ID '{$videoId}' is invalid");
                }

                $videoInfo = $videoInfoResponse['items'][0];

                $result[] = array(
                    'videoId' => $videoId,
                    'title' => $videoInfo['snippet']['title'],
                    'duplicate' => (bool) $this->findVideoPost($videoId),
                );
            }
        }

        return $result;
    }

    protected function _initCounts()
    {
        $this->_counts = array(
            'published' => 0,
            'processed' => 0,
            'duplicate' => 0,
            'filtered' => 0,
            'error' => 0,
        );
    }

    protected function _generatePostWithProgress($videoId)
    {
        $this->_counts['processed']++;

        try {
            $this->generatePost($videoId);
        } catch (ytg_Component_Generator_DuplicateException $e) {
            $this->_counts['duplicate']++;
            return;
        } catch (ytg_Component_Generator_FilteredException $e) {
            $this->_counts['filtered']++;
            return;
        } catch (ytg_Component_Generator_Exception $e) {
            $this->_counts['error']++;
            return;
        }

        $this->_counts['published']++;
    }

    public function generatePost($videoId)
    {
        $displayId = "youtu.be/{$videoId}";

        $this->feedback("Processing video {$displayId}... ");

        // Skip already published posts
        $this->_checkVideoPublication($videoId);

        // Get video info
        $videoInfoResponse = ytg_Core::$app->client_Youtube->request('videos', array(
            'part' => $this->_videoInfoParts,
            'id' => $videoId,
        ));

        if (!isset($videoInfoResponse['items'][0])) {
            $this->feedback("error: video ID '{$videoId}' is invalid.\n");
            throw new ytg_Component_Generator_Exception("Video ID '{$videoId}' is invalid");
        }

        if (ytg_Core::$app->options->lock) {
            return NULL;
        }

        $videoInfo = $videoInfoResponse['items'][0];

        // Post-search filter
        if ($this->_enableCheckVideoInfo) {
            $this->_checkVideoInfo($videoInfo);
        }

        // Create item array
        $snippet = $videoInfo['snippet'];
        $item = array(
            'videoId' => $videoId,
            'snippet' => $snippet,
            'title' => trim($snippet['title']),
            'content' => trim($snippet['description']),
            'tags' => isset($snippet['tags'])
                ? $snippet['tags']
                : array(),
            'postTemplate' => $this->postTemplate,
        );

        if ($this->generatorRemoveLinks) {
            $item['content'] = $this->_removeLinks($item['content']);
        }

        $this->feedback("ok, video title is \"{$item['title']}\".\n");

        // Text spinning
        if (ytg_Core::$app->client_Spinrewriter->enabled) {
            $this->_spinText($item);
        }

        if (!ytg_Core::$app->activate->activated) {
            return NULL;
        }

        // Publishing
        $result = $this->_publish($item);

        // Sharing
        if (ytg_Core::$app->client_Onlywire->enabled) {
            $this->_sharePostDeferred($result);
        }

        return $result;
    }

    public function __get($name)
    {
        if ($this->canGetProperty($name)) {
            return parent::__get($name);
        }

        return $this->getOption($name);
    }

    public function getOption($name)
    {
        if (array_key_exists($name, $this->_forceOptions)) {
            return $this->_forceOptions[$name];
        }

        if (array_key_exists($name, $this->options)) {
            return $this->options[$name];
        }

        return ytg_Core::$app->options->get($name);
    }

    public function feedback($options = array())
    {
        if ($this->process) {
            $this->process->feedback($options);
        } else if (!is_null($this->log)) {
            $log = NULL;
            if (is_array($options) && isset($options['log'])) {
                $log = $options['log'];
            } else {
                $log = $options;
            }

            if (!is_null($log)) {
                $this->log .= $log;
            }
        }
    }

    protected function _prepareSearchParams()
    {
        $result = array(
            'type' => 'video',
            'maxResults' => $this->searchMaxResults,
            'q' => $this->_prepareKeyword($this->keyword),
            'order' => $this->sortOrder,
        );

        $value = $this->filterDuration;
        if ('' != $value) {
            $result['videoDuration'] = $value;
        }

        if ('' != $result['q'] && '' != $this->negativeKeywords) {
            $result['q'] .= ' ' . $this->_prepareNegativeKeywords(
                $this->negativeKeywords);
        }

        return $result;
    }

    protected function _prepareKeyword($string)
    {
        if (!$this->channels) {
            return $string;
        }

        return preg_replace('~(?:"[^"]++")(*SKIP)(*F)|\s+~isu', '|',
            trim($string));
    }

    protected function _prepareNegativeKeywords($list)
    {
        $keywords = explode("\n", $list);
        foreach ($keywords as &$keyword) {
            if (FALSE !== strpos($keyword, ' ')) {
                // Multi word keyword
                $keyword = '"' . $keyword . '"';
            }

            $keyword = '-' . $keyword;
        }

        return implode(' ', $keywords);
    }

    protected function _initCheckChannelsSearchParams()
    {
        $this->_enableCheckVideoInfo = FALSE;

        return array(
            'type' => 'video',
            'maxResults' => $this->searchMaxResults,
            'order' => 'date',
        );
    }

    protected function _prepareChannelSearchParams(ytg_Model_Channel $channel)
    {
        $result = array(
            'channelId' => $channel->youtube_id,
            'publishedAfter' => gmdate('Y-m-d\Th:i:s\Z',
                $channel->last_check_timestamp),
            'q' => $channel->keyword,
        );

        if ('' != $result['q'] && '' != $channel->negative_keywords) {
            $result['q'] .= ' ' . $this->_prepareNegativeKeywords(
                $channel->negative_keywords);
        }

        return $result;
    }

    public function _prepareCheckVideoInfo()
    {
        if (!$this->videoIds && $this->minViews) {
            $result = array('snippet');
            $result[] = 'statistics';

            $this->_videoInfoParts = implode(',', $result);

            $this->_enableCheckVideoInfo = TRUE;
        }

        return $this;
    }

    protected function _checkVideoInfo(array $videoInfo)
    {
        if (isset($videoInfo['statistics']['viewCount'])) {
            $viewCount = $videoInfo['statistics']['viewCount'];
            if ($viewCount < $this->minViews) {
                $this->feedback("view count of {$viewCount} is less than required {$this->minViews}. Skipping.\n");
                throw new ytg_Component_Generator_FilteredException("View count of {$viewCount} is less than required {$this->minViews}");
            }
        }
    }

    public function findVideoPost($videoId)
    {
        $result = get_posts(array(
            'meta_key' => ytg_Core::$app->prefix . '_videoId',
            'meta_value' => $videoId,
            'posts_per_page' => 1,
            'orderby' => 'none',
            'post_status' => array('publish', 'pending', 'draft', 'future'),
            'post_type' => 'any',
        ));

        if ($result) {
            if (!isset($result[0])) {
                throw new Exception('Invalid get_posts() result');
            }

            return $result[0];
        }

        return NULL;
    }

    protected function _checkVideoPublication($videoId)
    {
        $post = $this->findVideoPost($videoId);

        if  ($post) {
            /**
             * @var WP_Post $result
             */

            $url = get_permalink($post->ID);

            $this->feedback("a post for this video already exists: {$url}. Skipping.\n");
            throw new ytg_Component_Generator_DuplicateException("A post for this video already exists: {$url}");
        }
    }

    protected function _removeLinks($text)
    {
        return preg_replace('~\b(https?|ftp|file)://[-A-Z0-9+&@#/%?=\~_|$!:,.;]*[A-Z0-9+&@#/%=\~_|$] *~isu', ' ', $text);
    }

    protected function _getThumbnailUrl($snippet)
    {
        $keys = array('maxres', 'standard', 'high', 'medium', 'default');

        foreach ($keys as $key) {
            if (isset($snippet['thumbnails'][$key]['url'])) {
                return $snippet['thumbnails'][$key]['url'];
            }
        }

        return NULL;
    }

    protected function _spinText(&$item)
    {

        try {
            $item['content'] = ytg_Core::$app->client_Spinrewriter->spin(
                $item['content'],
                array(
                    'protected_terms' => array($item['snippet']['title'])
                )
            );
        } catch (ytg_Component_Client_Spinrewriter_QuotaException $e) {
            $this->feedback("failed: out of quota.\n");

            if (!$this->allowTextSpinningOff) {
                throw new ytg_Component_Process_FatalException("Text rotation quota exceeded: {$e->getMessage()}");
            }

        } catch (ytg_Component_Client_Spinrewriter_Exception $e)  {
            throw new ytg_Component_Process_FatalException("Text rotation failed: {$e->getMessage()}");
        }
    }

    protected function _publish(&$item)
    {
        $this->feedback("Saving the post... ");

        /**
         * @var ytg_Component_Generator_Publisher $publisher
         */
        $publisher = ytg_Core::$app->generator_Publisher;

        try {
            $postId = $publisher->publish($item);
        } catch (ytg_Component_Generator_Publisher_Exception $e) {
            throw new ytg_Component_Process_FatalException("Post creation failed: {$e->getMessage()}");
        }

        $result = get_post($postId);

        $this->feedback("ok.\n");

        // Featured image
        if ($this->generatorFeaturedImages) {
            $this->feedback("Adding featured image... ");

            $url = $this->_getThumbnailUrl($item['snippet']);
            if (!is_null($url)) {
                try {
                    $publisher->downloadFeaturedImage($url, $item);
                } catch (ytg_Component_Generator_Publisher_Exception $e) {
                    throw new ytg_Component_Process_FatalException("Error adding featured image form {$url}: {$e->getMessage()}");
                }
            } else {
                $this->feedback("no video thumbnails found.\n");
            }

            $this->feedback("ok.\n");
        }

        // Comments
        if ($this->generatorFetchComments) {
            $publisher->addComments($item, $result);
        }

        return $result;
    }

    protected function _sharePostDeferred(WP_Post $post)
    {
        $shared = FALSE;

        if ('publish' == $post->post_status) {
            $this->feedback("Sharing the post... ");

            try {
                $this->sharePost($post);
                $shared = TRUE;
            } catch (ytg_Component_Generator_ShareException $e) {
                $this->feedback("error: {$e->getMessage()}\n");
            }
        } else {
            $this->feedback("Preparing the post for sharing on publication... ");
        }

        if (!$shared) {
            $meta = ytg_Core::create('Framework_Component_Meta', array(
                'id' => $post->ID,
            ));

            $meta->set('onlywireServices',
                ytg_Core::$app->client_Onlywire->selectedServices);
        }

        $this->feedback("ok.\n");
    }

    public function sharePost($post, $services=NULL)
    {
        /**
         * @var ytg_Framework_Component_Meta $meta
         */
        $meta = ytg_Core::create('Framework_Component_Meta', array(
            'id' => $post->ID,
        ));

        try {
            ytg_Core::$app->client_Onlywire->addPost($post, $services);
        } catch (ytg_Component_Client_Onlywire_Exception $e) {
            $msg = $e->getMessage();
            $meta->set('onlywireError', $msg);
            throw new ytg_Component_Generator_ShareException($msg);
        }

        $meta->delete('onlywireError');
    }

    public function checkChannels()
    {
        $this->log = '';
        $this->feedback("Starting... ");

        $this->_forceOptions = array(
            'publicationMethod' => 'publish',
        );

        $channels = ytg_Core::$app->model('Model_Channel')->findAll();

        if (!$channels) {
            $this->feedback("no channels found.");
            return;
        }

        /**
         * @var ytg_Component_Client_Youtube $youtube
         */
        $youtube = ytg_Core::$app->client_Youtube;
        $commonSearchParams = $this->_initCheckChannelsSearchParams();

        $this->_initCounts();

        $this->feedback("channels to check: " . count($channels) . ".\n\n");

        /**
         * @var ytg_Model_Channel $channel
         */
        foreach ($channels as $channel) {
            $this->feedback("Checking channel {$channel->name} ({$channel->youtube_id})... ");

            $videoIds = array();

            $searchParams = array_merge($commonSearchParams,
                $this->_prepareChannelSearchParams($channel));

            $channel->last_check_timestamp = time();

            do {
                // Search request
                try {
                    $response = $youtube->search($searchParams);
                } catch (ytg_Component_Client_Youtube_Exception $e) {
                    throw new ytg_Component_Process_FatalException(
                        "YouTube API request failed: {$e->getMessage()}");
                }

                if (!$response['items'] && !isset($searchParams['pageToken'])) {
                    break;
                }

                foreach ($response['items'] as $videoInfo) {
                    $videoIds[] = $videoInfo['id']['videoId'];
                }

                if (isset($response['nextPageToken'])) {
                    $searchParams['pageToken'] = $response['nextPageToken'];
                } else {
                    unset ($searchParams['pageToken']);
                }
            } while (isset($response['nextPageToken']));

            if (!$videoIds) {
                $channel->save(FALSE);
                $this->feedback("no new videos found.\n\n");
                continue;
            }

            // Set up publisher
            $this->_forceOptions['postCategories'] = $channel->categories;
            $this->_forceOptions['postType'] = $channel->post_type;

            $this->feedback("found new videos: " . count($videoIds) . ".\n\n");

            $videoIds = array_reverse($videoIds);

            foreach ($videoIds as $videoId) {
                $this->_generatePostWithProgress($videoId);
                // Extra line break at the end of post generation
                $this->feedback("\n");
            }

            $channel->save(FALSE);
        }

        $this->feedback("Finished. Generated posts: {$this->_counts['published']}. "
            . "Processed videos: {$this->_counts['processed']}, "
            . "already published: {$this->_counts['duplicate']}.");
    }
}

class ytg_Component_Generator_Exception extends Exception
{

}

class ytg_Component_Generator_FilteredException
    extends ytg_Component_Generator_Exception
{

}

class ytg_Component_Generator_DuplicateException
    extends ytg_Component_Generator_Exception
{

}

class ytg_Component_Generator_ShareException
    extends ytg_Component_Generator_Exception
{

}