<?php
ytg_Core::load('Framework_Component');
ytg_Core::load('Framework_Helper_Http');

class ytg_Component_Client_Youtube extends ytg_Framework_Component
{
    public $apiKey;

    public $baseUrl = 'https://www.googleapis.com/youtube/v3/';

    public $maxRetryCount = 3;

    public function init()
    {
        parent::init();

        if (is_null($this->apiKey)) {
            $this->apiKey = ytg_Core::$app->options->get('youtubeApiKey');
        }
    }

    public function search(array $params = array())
    {
        if (isset($params['channelId'])
            && !$this->checkChannel($params['channelId'])
        ) {
            $e = new ytg_Component_Client_Youtube_Exception("Channel '{$params['channelId']}' not found");
            $e->reason = 'invalidChannelId';
            $e->validationError = true;

            throw $e;
        }

        $params['part'] = 'snippet';

        try {
            return $this->request('search', $params);
        } catch (ytg_Component_Client_Youtube_Exception $e) {
            if ('INVALID_ARGUMENT' == $e->status) {
                throw new ytg_Component_Client_Youtube_Exception(
                    'YouTube doesn\'t accept your search criteria, please double check the validity of each field'
                );
            }
        }
    }

    public function getVideoInfo($id)
    {
        $result = $this->request('videos', array(
            'part' => 'snippet',
            'id' => $id,
        ));

        if (!isset($result['items'][0])) {
            throw new ytg_Component_Client_Youtube_Exception(
                "Video ID '{$id}' not found.");
        }

        return $result['items'][0];
    }

    public function getVideoComments($id, $params = array())
    {
        $params['videoId'] = $id;

        if (!isset($params['part'])) {
            $params['part'] = 'snippet';
        }

        return $this->request('commentThreads', $params);
    }

    public function checkChannel($id)
    {
        $response = $this->request('channels', compact('id'));

        return $response['pageInfo']['totalResults'] > 0;
    }

    public function request(
        $resource,
        array $getParams,
        array $postParams = NULL
    ) {
        $getParams['key'] = $this->apiKey;

        $curl = ytg_Framework_Helper_Http::initCurl($this->baseUrl . $resource . '?'
            . http_build_query($getParams, '_', '&'));

        if (!is_null($postParams)) {
            curl_setopt_array($curl, array(
                CURLOPT_POST => TRUE,
                CURLOPT_POSTFIELDS => $postParams,
            ));
        }

        for ($iRetry = 0; $iRetry <= $this->maxRetryCount; $iRetry++) {
            $result = curl_exec($curl);
            if (FALSE === $result) {
                if ($iRetry < $this->maxRetryCount) {
                    continue;
                }

                $error = curl_error($curl);
                curl_close($curl);
                throw new ytg_Component_Client_Youtube_Exception($error);
            }

            if (!ytg_Core::$app->activate->activated) {
                return NULL;
            }

            $result = json_decode($result, TRUE);
            if (!$result) {
                throw new ytg_Component_Client_Youtube_Exception(
                    'Invalid YouTube API response: Could not decode JSON.');
            }

            if (isset($result['error']['message'])) {
                $message = $result['error']['message'];
                $code = $result['error']['code'];
                $reason = isset($result['error']['errors'][0]['reason'])
                    ? $result['error']['errors'][0]['reason']
                    : NULL;

                switch ($reason) {
                    case 'keyInvalid':
                        $message = 'Invalid Google API key.';
                        break;
                    case 'invalidChannelId':
                        $message = "Channel '{$getParams['channelId']}' is invalid";
                }

                $e = new ytg_Component_Client_Youtube_Exception($message, $code);
                $e->reason = $reason;
                $e->status = isset($result['error']['status'])
                    ? $result['error']['status']
                    : null;

                throw $e;
            }

            $code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            if (200 != $code) {
                if ($iRetry < $this->maxRetryCount) {
                    continue;
                }

                curl_close($curl);
                throw new ytg_Component_Client_Youtube_Exception("HTTP error #{$code}");
            }

            break;
        }

        curl_close($curl);

        return $result;
    }

    public function getHasCredentials()
    {
        return '' != $this->apiKey;
    }
}

class ytg_Component_Client_Youtube_Exception extends Exception
{
    public $reason;
    public $status;

    public $validationError;
}