<?php
ytg_Core::load('Framework_Component');
ytg_Core::load('Framework_Helper_Http');

class ytg_Component_Client_Spinrewriter extends ytg_Framework_Component
{
    public $email;
    public $apiKey;

    public $apiUrl = 'http://www.spinrewriter.com/action/api';

    public $spinTimeout = 7;
    public $retryCount = 3;
    public $retryDelay = 5;

    public $maxRetryCount = 3;

    public $minTextLength = 10;

    public $maxWordCount = 4000;

    protected $_enabled;

    protected $_lastSpinTimestamp;

    public $commonSpinParams = array(
        'auto_protected_terms' => 'true',
    );

    public function init()
    {
        parent::init();

        if (is_null($this->email)) {
            $this->email = ytg_Core::$app->options->get('spinrewriterEmail');
            $this->apiKey = ytg_Core::$app->options->get('spinrewriterApiKey');
        }

        $this->_enabled = ('' != $this->email)
            && ('' != $this->apiKey)
            && !ytg_Core::$app->options->get('spinrewriterDisable');
    }

    public function spin($text, array $params = array())
    {
        if (!$this->_enabled) {
            return $text;
        }

        if (strlen($text) < $this->minTextLength) {
            ytg_Core::$app->generator->feedback("Text is too short for spinning.\n");

            return $text;
        }

        $wordCount = str_word_count($text);
        if ($wordCount > $this->maxWordCount) {
            ytg_Core::$app->generator->feedback("Text is too long for spinning: {$wordCount} words. Maximum {$this->maxWordCount} words allowed by SpinRewriter.\n");

            return $text;
        }

        if (!is_null($this->_lastSpinTimestamp)) {
            $delta = ceil($this->_lastSpinTimestamp + $this->spinTimeout
                - time());
            if ($delta > 0) {
                ytg_Core::$app->generator->feedback("Wait {$delta} sec before the next SpinRewriter request... ");
                sleep($delta);
                ytg_Core::$app->generator->feedback("done.\n");
            }
        }

        ytg_Core::$app->generator->feedback("Spinning the text... ");

        $protectedTerms = ytg_Core::$app->options->get('spinrewriterProtectedTerms');
        $protectedTerms = '' != $protectedTerms
            ? explode("\n", $protectedTerms)
            : array();

        if (isset($params['protected_terms'])) {
            $protectedTerms = array_merge($params['protected_terms'],
                $protectedTerms);
        }
        if ($protectedTerms) {
            $params['protected_terms'] = implode("\n", $protectedTerms);
        }

        if (!isset($params['confidence_level'])) {
            $params['confidence_level'] =
                ytg_Core::$app->options->get('spinrewriterConfidenceLevel');
        }

        $params['text'] = $text;

        try {
            $result = $this->request('unique_variation', $params);
        } catch (ytg_Component_Client_Spinrewriter_TextTooShortException $e) {
            ytg_Core::$app->generator->feedback("failed: text is too short.\n");
            return $text;
        } catch (ytg_Component_Client_Spinrewriter_TextTooLongException $e) {
            ytg_Core::$app->generator->feedback("failed: text is too long.\n");
            return $text;
        }

        $this->_lastSpinTimestamp = time();

        ytg_Core::$app->generator->feedback("ok.\n");

        return $result['response'];
    }

    public function getQuota()
    {
        $result = $this->request('api_quota');

        return (int) $result['api_requests_available'];
    }

    public function request($action, array $params = array())
    {
        if (!$this->_enabled) {
            throw new ytg_Component_Client_Spinrewriter_Exception(
                'Spin Rewriter credentials are not set up.');
        }

        $retryDelay = 'api_quota' != $action;

        $params['action'] = $action;
        $params['email_address'] = $this->email;
        $params['api_key'] = $this->apiKey;

        $curl = ytg_Framework_Helper_Http::initCurl($this->apiUrl, array(
            CURLOPT_POST => TRUE,
            CURLOPT_POSTFIELDS => $params,
        ));

        for ($iRetry = 0; $iRetry <= $this->retryCount; $iRetry++) {
            $result = curl_exec($curl);
            if (FALSE === $result) {
                if ($iRetry < $this->maxRetryCount) {
                    continue;
                }

                $error = curl_error($curl);
                curl_close($curl);
                throw new ytg_Component_Client_Spinrewriter_Exception($error);
            }

            $result = json_decode($result, TRUE);
            if (!$result) {
                if ($iRetry < $this->maxRetryCount) {
                    if ($retryDelay) {
                        $this->_delayBeforeRetry();
                    }
                    continue;
                }

                if (strlen($result) > 512) {
                    $result = substr($result, 0, 512) . '...';
                }

                throw new ytg_Component_Client_Spinrewriter_Exception(
                    "Invalid Spin Rewriter API response: '{$result}'");
            }

            if ('OK' != $result['status']) {
                if ('Original text too short.' == $result['response']) {
                    throw new ytg_Component_Client_Spinrewriter_TextTooShortException(
                        "Spin Rewriter API returned error: {$result['response']}");
                }
                if ('Original text too long.' == substr($result['response'], 0, 23)) {
                    throw new ytg_Component_Client_Spinrewriter_TextTooLongException(
                        "Spin Rewriter API returned error: {$result['response']}");
                }
                if (substr($result['response'], 0, 19) == 'API quota exceeded.') {
                    $this->_enabled = FALSE;
                    throw new ytg_Component_Client_Spinrewriter_QuotaException(
                        "Spin Rewriter API returned error: {$result['response']}");
                }

                if ($iRetry < $this->retryCount
                    && (!isset($result['api_requests_available'])
                        || $result['api_requests_available'] > 0
                    )
                ) {
                    if ($retryDelay) {
                        $this->_delayBeforeRetry();
                    }
                    continue;
                }

                throw new ytg_Component_Client_Spinrewriter_Exception(
                    "Spin Rewriter API returned error: {$result['response']}");
            }

            $code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            if (200 != $code) {
                if ($iRetry < $this->maxRetryCount) {
                    continue;
                }

                curl_close($curl);
                throw new ytg_Component_Client_Spinrewriter_Exception("HTTP error #{$code}");
            }

            break;
        }

        curl_close($curl);
        return $result;
    }

    protected function _delayBeforeRetry()
    {
        ytg_Core::$app->generator->feedback("(wait {$this->retryDelay} sec due to error... ");

        sleep($this->retryDelay);

        ytg_Core::$app->generator->feedback("done) retrying... ");
    }

    public function getEnabled()
    {
        return $this->_enabled;
    }
}

class ytg_Component_Client_Spinrewriter_Exception extends Exception
{

}

class ytg_Component_Client_Spinrewriter_TextTooShortException
    extends ytg_Component_Client_Spinrewriter_Exception
{

}

class ytg_Component_Client_Spinrewriter_TextTooLongException
    extends ytg_Component_Client_Spinrewriter_Exception
{

}

class ytg_Component_Client_Spinrewriter_QuotaException
    extends ytg_Component_Client_Spinrewriter_Exception
{

}