<?php
ytg_Core::load('Framework_Component');
ytg_Core::load('Framework_Helper_Http');

class ytg_Component_Client_Onlywire extends ytg_Framework_Component
{
    public $username;
    public $password;

    public $baseUrl = 'https://www.onlywire.com/api/';

    public $serviceCacheKeyPrefix = 'onlywire_services_';
    public $serviceCacheExpiration = 36000;

    public $selectedServices;
    public $services;

    public $maxRetryCount = 3;

    public $maxContentLength = 4096;

    public function init()
    {
        parent::init();

        if (is_null($this->username)) {
            $this->username = ytg_Core::$app->options->get('onlywireUsername');
            $this->password = ytg_Core::$app->options->get('onlywirePassword');
        }

        if (is_null($this->selectedServices)) {
            $this->selectedServices = ytg_Core::$app->options->get('onlywireServices');
        }
    }

    public function addPost(WP_Post $post, $services = NULL)
    {
        $content = trim($post->post_content);
        if (strlen($content) > $this->maxContentLength || '' == $content) {
            $content = $post->post_title;
        }

        return $this->add(array(
            'url' => get_permalink($post->ID),
            'title' => $post->post_title,
            'comments' => $content,
            'tags' => implode(',', wp_get_post_tags($post->ID, array(
                'fields' => 'names')
            ))),
            $services
        );
    }

    public function add(array $params, $services = NULL)
    {
        if (!$this->enabled) {
            throw new ytg_Component_Client_Onlywire_Exception(
                'OnlyWire credentials are not set up.');
        }

        if (is_null($services)) {
            $services = $this->selectedServices;
        }

        if ($services) {
            $allServices = $this->getServiceIds();
            $services = array_intersect($services, $allServices);

            $params['service_logins'] = implode(',', $services);
        }

        $result = $this->request('add', $params);

        if (!isset($result['result'])) {
            throw new ytg_Component_Client_Onlywire_Exception(
                'Invalid API response: result field not found.');
        }

        if ('success' != $result['result']) {
            $message = $result['result'];
            $message = preg_replace('~^error\:\s*~isu', '', $message);

            if ('failed' == $message) {
                $message = 'Invalid login/password.';
            }

            throw new ytg_Component_Client_Onlywire_Exception($message);
        }

        return $this;
    }

    public function request($command, array $params = array())
    {
        $params['type'] = 'json';

        $curl = ytg_Framework_Helper_Http::initCurl($this->baseUrl . $command . '?'
            . http_build_query($params, '_', '&'),
            array(
                CURLOPT_USERPWD => "{$this->username}:{$this->password}",
            )
        );

        for ($iRetry = 0; $iRetry <= $this->maxRetryCount; $iRetry++) {
            $result = curl_exec($curl);
            if (FALSE === $result) {
                if ($iRetry < $this->maxRetryCount) {
                    continue;
                }

                $error = curl_error($curl);
                curl_close($curl);
                throw new ytg_Component_Client_Onlywire_Exception($error);
            }

            if ('' == $result || 'failed' == $result) {
                curl_close($curl);
                throw new ytg_Component_Client_Onlywire_Exception(
                    'Invalid OnlyWire credentials.');
            }

            $code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            if (200 != $code) {
                if ($iRetry < $this->maxRetryCount) {
                    continue;
                }

                curl_close($curl);
                throw new ytg_Component_Client_Onlywire_Exception("HTTP error #{$code}");
            }

            break;
        }

        curl_close($curl);

        if ('[]' == $result) {
            return array();
        }

        $result = json_decode($result, TRUE);
        if (!$result) {
            throw new ytg_Component_Client_Onlywire_Exception(
                'Invalid API response: Could not decode JSON.');
        }

        return $result;
    }

    public function getServices()
    {
        if ($this->services) {
            return $this->services;
        }

        $cacheKey = $this->getServiceCacheKey();

        $result = ytg_Core::$app->cache->get($cacheKey);

        if (is_null($result)) {
            $result = $this->request('services');

            ytg_Core::$app->cache->set($cacheKey, $result,
                $this->serviceCacheExpiration);
            $this->services = $result;
        }

        return $result;
    }

    public function resetServiceCache()
    {
        ytg_Core::$app->cache->delete($this->getServiceCacheKey());
        $this->services = NULL;

        return $this;
    }

    public function getServiceList()
    {
        $result = array();
        foreach ($this->getServices() as $service) {
            $result[$service['service_login_id']]
                = "{$service['name']}: {$service['account_info']}";
        }

        return $result;
    }

    public function getServiceListGrouped()
    {
        $result = array();
        foreach ($this->getServices() as $service) {
            $groupName = $service['name_clean'];
            if (!isset($result[$groupName])) {
                $result[$groupName] = array(
                    'name' => $groupName,
                    'label' => $service['name'],
                    'items' => array(),
                );
            }

            $result[$groupName]['items'][$service['service_login_id']]
                = $service['account_info'];
        }

        return $result;
    }

    public function getServiceIds()
    {
        $response = $this->getServices();

        $result = array();
        foreach ($response as $service) {
            $result[] = $service['service_login_id'];
        }

        return $result;
    }

    public function getServiceCacheKey()
    {
        return $this->serviceCacheKeyPrefix . $this->username;
    }

    public function getHasCredentials()
    {
        return '' != $this->username;
    }

    public function getEnabled()
    {
        return $this->getHasCredentials()
        && (is_null($this->selectedServices) || $this->selectedServices);
    }
}

class ytg_Component_Client_Onlywire_Exception extends Exception
{

}