<?php
ytg_Core::load('Framework_Component');

abstract class ytg_Framework_Model extends ytg_Framework_Component
{
    const SCENARIO_DEFAULT = 'default';

    /**
     * @var array validation errors (attribute name => array of errors)
     */
    private $_errors;
    /**
     * @var ArrayObject list of validators
     */
    private $_validators;
    /**
     * @var string current scenario
     */
    private $_scenario = self::SCENARIO_DEFAULT;

    static protected $_models = array();

    public $attributeProperties = array();

    public $attributePropertyDefaults = array();

    public function __construct($config=array())
    {
        // Do not initialize is called from model() method
        if (!is_null($config)) {
            parent::__construct($config);
        }
    }

    public static function model($className)
    {
        if (isset(self::$_models[$className])) {
            return self::$_models[$className];
        }

        $result = new $className(NULL);
        self::$_models[$className] = $result;

        return $result;
    }

    public function rules()
    {
        return array();
    }

    public function scenarios()
    {
        $scenarios = array(self::SCENARIO_DEFAULT => array());
        foreach ($this->getValidators() as $validator) {
            foreach ($validator->on as $scenario) {
                $scenarios[$scenario] = array();
            }
            foreach ($validator->except as $scenario) {
                $scenarios[$scenario] = array();
            }
        }
        $names = array_keys($scenarios);

        foreach ($this->getValidators() as $validator) {
            if (empty($validator->on) && empty($validator->except)) {
                foreach ($names as $name) {
                    foreach ($validator->attributes as $attribute) {
                        $scenarios[$name][$attribute] = true;
                    }
                }
            } elseif (empty($validator->on)) {
                foreach ($names as $name) {
                    if (!in_array($name, $validator->except, true)) {
                        foreach ($validator->attributes as $attribute) {
                            $scenarios[$name][$attribute] = true;
                        }
                    }
                }
            } else {
                foreach ($validator->on as $name) {
                    foreach ($validator->attributes as $attribute) {
                        $scenarios[$name][$attribute] = true;
                    }
                }
            }
        }

        foreach ($scenarios as $scenario => $attributes) {
            if (empty($attributes) && $scenario !== self::SCENARIO_DEFAULT) {
                unset($scenarios[$scenario]);
            } else {
                $scenarios[$scenario] = array_keys($attributes);
            }
        }

        return $scenarios;
    }

    /**
     * @return string
     */
    public function formName()
    {
        $reflector = new ReflectionClass($this);

        return $reflector->getName();
    }

    public function attributes()
    {
        $class = new ReflectionClass($this);
        $names = array();
        foreach ($class->getProperties(ReflectionProperty::IS_PUBLIC) as $property) {
            if (!$property->isStatic()) {
                $names[] = $property->getName();
            }
        }

        return $names;
    }

    public function attributeLabels()
    {
        return array();
    }

    public function attributeCamelNames()
    {
        return array();
    }

    public function validate($attributeNames = null, $clearErrors = true)
    {
        if ($clearErrors) {
            $this->clearErrors();
        }

        if (!$this->beforeValidate()) {
            return false;
        }

        $scenarios = $this->scenarios();
        $scenario = $this->getScenario();
        if (!isset($scenarios[$scenario])) {
            throw new Exception("Unknown scenario: $scenario");
        }

        if ($attributeNames === null) {
            $attributeNames = $this->activeAttributes();
        }

        foreach ($this->getActiveValidators() as $validator) {
            $validator->validateAttributes($this, $attributeNames);
        }
        $this->afterValidate();

        return !$this->hasErrors();
    }

    /**
     * This method is invoked before validation starts.
     * The default implementation raises a `beforeValidate` event.
     * You may override this method to do preliminary checks before validation.
     * Make sure the parent implementation is invoked so that the event can be raised.
     * @return boolean whether the validation should be executed. Defaults to true.
     * If false is returned, the validation will stop and the model is considered invalid.
     */
    public function beforeValidate()
    {
        return TRUE;
    }

    /**
     * This method is invoked after validation ends.
     * The default implementation raises an `afterValidate` event.
     * You may override this method to do postprocessing after validation.
     * Make sure the parent implementation is invoked so that the event can be raised.
     */
    public function afterValidate()
    {
        // Extensions
    }

    public function getValidators()
    {
        if ($this->_validators === null) {
            $this->_validators = $this->createValidators();
        }
        return $this->_validators;
    }

    public function getActiveValidators($attribute = null)
    {
        $validators = array();
        $scenario = $this->getScenario();
        foreach ($this->getValidators() as $validator) {
            if ($validator->isActive($scenario) && ($attribute === null || in_array($attribute, $validator->attributes, true))) {
                $validators[] = $validator;
            }
        }
        return $validators;
    }

    public function createValidators()
    {
        ytg_Core::load('Framework_Validator');

        $validators = array();
        foreach ($this->rules() as $rule) {
            if ($rule instanceof ytg_Framework_Validator) {
                $validators[] = $rule;
            } elseif (is_array($rule) && isset($rule[0], $rule[1])) { // attributes, validator type
                $validator = ytg_Framework_Validator::createValidator($rule[1], $this, (array) $rule[0], array_slice($rule, 2));
                $validators[] = $validator;
            } else {
                throw new Exception('Invalid validation rule: a rule must specify both attribute names and validator type.');
            }
        }
        return $validators;
    }

    public function isAttributeRequired($attribute)
    {
        foreach ($this->getActiveValidators($attribute) as $validator) {
            if ($validator instanceof ytg_Framework_Validator_Required && $validator->when === null) {
                return true;
            }
        }
        return false;
    }

    public function isAttributeSafe($attribute)
    {
        return in_array($attribute, $this->safeAttributes(), true);
    }

    public function isAttributeActive($attribute)
    {
        return in_array($attribute, $this->activeAttributes(), true);
    }

    public function getAttributeLabel($attribute)
    {
        $labels = $this->attributeLabels();
        return isset($labels[$attribute]) ? $labels[$attribute] : $this->generateAttributeLabel($attribute);
    }

    public function hasErrors($attribute = null)
    {
        return $attribute === null ? !empty($this->_errors) : isset($this->_errors[$attribute]);
    }

    public function getErrors($attribute = null)
    {
        if ($attribute === null) {
            return $this->_errors === null ? array() : $this->_errors;
        } else {
            return isset($this->_errors[$attribute]) ? $this->_errors[$attribute] : array();
        }
    }

    public function getFirstErrors()
    {
        if (empty($this->_errors)) {
            return array();
        } else {
            $errors = array();
            foreach ($this->_errors as $name => $es) {
                if (!empty($es)) {
                    $errors[$name] = reset($es);
                }
            }

            return $errors;
        }
    }

    public function getFirstError($attribute)
    {
        return isset($this->_errors[$attribute]) ? reset($this->_errors[$attribute]) : null;
    }

    public function addError($attribute, $error = '')
    {
        $this->_errors[$attribute][] = $error;
    }

    public function addErrors(array $items)
    {
        foreach ($items as $attribute => $errors) {
            if (is_array($errors)) {
                foreach($errors as $error) {
                    $this->addError($attribute, $error);
                }
            } else {
                $this->addError($attribute, $errors);
            }
        }
    }

    public function clearErrors($attribute = null)
    {
        if ($attribute === null) {
            $this->_errors = array();
        } else {
            unset($this->_errors[$attribute]);
        }
    }

    public function generateAttributeLabel($name)
    {
        $result = '';

        $length = strlen($name);
        for ($i=0; $i < $length; $i++) {
            if ($i > 0 && ctype_upper($name[$i])) {
                $result .= ' ';
            }

            $result .= $name[$i];
        }

        $result = ucfirst($result);

        return $result;
    }

    public function getAttributes($names = null, $except = array())
    {
        $values = array();
        if ($names === null) {
            $names = $this->attributes();
        }
        foreach ($names as $name) {
            $values[$name] = $this->$name;
        }
        foreach ($except as $name) {
            unset($values[$name]);
        }

        return $values;
    }

    public function setAttributes($values, $safeOnly = true)
    {
        if (is_array($values)) {
            $attributes = array_flip($safeOnly ? $this->safeAttributes() : $this->attributes());
            foreach ($values as $name => $value) {
                if (isset($attributes[$name])) {
                    $this->$name = $value;
                } elseif ($safeOnly) {
                    $this->onUnsafeAttribute($name, $value);
                }
            }
        }
    }

    public function onUnsafeAttribute($name, $value)
    {
        // Nothing
    }

    public function getScenario()
    {
        return $this->_scenario;
    }

    public function setScenario($value)
    {
        $this->_scenario = $value;
    }

    public function safeAttributes()
    {
        $scenario = $this->getScenario();
        $scenarios = $this->scenarios();
        if (!isset($scenarios[$scenario])) {
            return array();
        }
        $attributes = array();
        foreach ($scenarios[$scenario] as $attribute) {
            if ($attribute[0] !== '!') {
                $attributes[] = $attribute;
            }
        }

        return $attributes;
    }

    public function activeAttributes()
    {
        $scenario = $this->getScenario();
        $scenarios = $this->scenarios();
        if (!isset($scenarios[$scenario])) {
            return array();
        }
        $attributes = $scenarios[$scenario];
        foreach ($attributes as $i => $attribute) {
            if ($attribute[0] === '!') {
                $attributes[$i] = substr($attribute, 1);
            }
        }

        return $attributes;
    }

    public function load($data, $formName = null)
    {
        $scope = $formName === null ? $this->formName() : $formName;
        if ($scope === '' && !empty($data)) {
            $this->setAttributes($data);

            return true;
        } elseif (isset($data[$scope]) && is_array($data[$scope])) {
            $this->setAttributes($data[$scope]);

            return true;
        } else {
            return false;
        }
    }

    public static function loadMultiple($models, $data, $formName = null)
    {
        if ($formName === null) {
            /* @var $first ytg_Framework_Model */
            $first = reset($models);
            if ($first === false) {
                return false;
            }
            $formName = $first->formName();
        }

        $success = false;
        foreach ($models as $i => $model) {
            /* @var $model ytg_Framework_Model */
            if ($formName == '') {
                if (!empty($data[$i])) {
                    $model->load($data[$i], '');
                    $success = true;
                }
            } elseif (!empty($data[$formName][$i])) {
                $model->load($data[$formName][$i], '');
                $success = true;
            }
        }

        return $success;
    }

    public static function validateMultiple($models, $attributeNames = null)
    {
        $valid = true;
        /* @var $model ytg_Framework_Model */
        foreach ($models as $model) {
            $valid = $model->validate($attributeNames) && $valid;
        }

        return $valid;
    }

    public function fields()
    {
        $fields = $this->attributes();

        return array_combine($fields, $fields);
    }

    public function getIterator()
    {
        $attributes = $this->getAttributes();
        return new ArrayIterator($attributes);
    }

    public function offsetExists($offset)
    {
        return $this->$offset !== null;
    }

    public function offsetGet($offset)
    {
        return $this->$offset;
    }

    public function offsetSet($offset, $item)
    {
        $this->$offset = $item;
    }

    public function offsetUnset($offset)
    {
        $this->$offset = null;
    }

    //
    // Custom mods
    //

    public function getAttributeCamelName($attribute)
    {
        $camelNames = $this->attributeCamelNames();

        if (isset($camelNames[$attribute])) {
            $attribute = $camelNames[$attribute];
        }

        return $attribute;
    }

    public function getAttributeList($attribute)
    {
        $attribute = $this->getAttributeCamelName($attribute);

        return $this->__get("{$attribute}List");
    }

    public function getListItemLabel($attribute, $value = FALSE)
    {
        if (FALSE === $value) {
            $value = $this->$attribute;
        }

        if ('' == $value) {
            return '';
        }

        $list = $this->getAttributeList($attribute);

        return isset($list[$value])? $list[$value] : "({$value})";
    }

    public function getAttributeProperty($attribute, $property, $default = NULL)
    {
        if (isset($this->attributeProperties[$attribute][$property])) {
            return $this->attributeProperties[$attribute][$property];
        }

        if (!is_null($default)) {
            return $default;
        }

        if (isset($this->attributePropertyDefaults[$property])) {
            return $this->attributePropertyDefaults[$property];
        }

        return NULL;
    }
}
