<?php
ytg_Core::load('Framework_Db_ActiveRecord');
ytg_Core::load('Framework_Helper_Filter');

/**
 * @property  int $id
 * @property string $name
 * @property string $youtube_id
 * @property int $last_check_timestamp
 * @property int $template_id
 * @property string $categories
 * @property ytg_Model_Template $template
 * @property string $keyword
 * @property string $negative_keywords
 * @property string $post_type
 */
class ytg_Model_Channel extends ytg_Framework_Db_ActiveRecord
{
    public $table = 'channel';

    protected $_categories;
    protected $_oldCategories;

    public function schema()
    {
        return array(
            'columns' => array(
                'id' => array(
                    'type' => 'integer',
                    'isPrimaryKey' => TRUE,
                    'autoIncrement' => TRUE,
                    'allowNull' => FALSE,
                ),
                'name' => array(
                    'type' => 'string',
                    'size' => 255,
                    'allowNull' => FALSE,
                ),
                'youtube_id' => array(
                    'type' => 'string',
                    'size' => 255,
                    'allowNull' => FALSE,
                ),
                'last_check_timestamp' => array(
                    'type' => 'integer',
                    'allowNull' => TRUE,
                ),
                'template_id' => array(
                    'type' => 'integer',
                    'allowNull' => TRUE,
                ),
                'keyword' => array(
                    'type' => 'string',
                    'allowNull' => TRUE,
                ),
                'negative_keywords' => array(
                    'type' => 'string',
                    'allowNull' => TRUE,
                ),
                'post_type' => array(
                    'type' => 'string',
                    'allowNull' => FALSE,
                ),
            ),
        );
    }

    public static function model($className=__CLASS__)
    {
        return parent::model($className);
    }

    //
    // Requisites
    //

    public function attributeCamelNames()
    {
        return array(
            'post_type' => 'postType',
        );
    }

    public function attributeLabels()
    {
        return array(
            'id' => 'ID',
            'name' => 'YouTube Channel Name',
            'youtube_id' => 'YouTube Channel ID',
            'template_id' => 'Post Template',
            'categories' => 'Publish Posts under This Category',
            'keyword' => 'Keyword',
            'negative_keywords' => 'Filter out These Keywords',
            'post_type' => 'Generate Custom Post Types',
        );
    }

    public function rules()
    {
        return array(
            array('name', 'filter', 'filter' => 'stripslashes'),
            array('name', 'trim'),
            array('name', 'filter',
                'filter' => array('ytg_Framework_Helper_Filter', 'stripNewLines')
            ),
            array('name', 'string', 'max' => 255),
            array('name', 'unique'),
            //array('name', 'required'),

            array('youtube_id', 'filter', 'filter' => 'stripslashes'),
            array('youtube_id', 'trim'),
            array('youtube_id', 'filter',
                'filter' => array('ytg_Framework_Helper_Filter', 'stripNewLines')
            ),
            array('youtube_id', 'filter',
                'filter' => array('Helper_YouTube', 'filterChannel'),
            ),
            array('youtube_id', 'string', 'max' => 255),
            array('youtube_id', 'required'),

            array('template_id', 'integer'),
            array('template_id', 'exist',
                'targetClass' => 'Model_Template',
                'targetAttribute' => 'id',
            ),
            array('template_id', 'default'),

            array('categories', 'Validator_PostCategories'),

            array('keyword', 'filter', 'filter' => 'stripslashes'),
            array('keyword', 'trim'),
            array('keyword', 'filter', 'filter'
                => array('ytg_Framework_Helper_Filter', 'stripNewLines')),
            array('keyword', 'match',
                'not' => TRUE,
                'pattern' => '~\-~isu',
                'message' => '{attribute} does not allow minus (-) character',
            ),
            array('keyword', 'match',
                'not' => TRUE,
                'pattern' => '~\|~isu',
                'message' => '{attribute} does not allow pipe (|) character',
            ),
            array('keyword', 'default'),

            array('negative_keywords', 'filter', 'filter' => 'stripslashes'),
            array('negative_keywords', 'trim'),
            array('negative_keywords', 'filter', 'filter'
                => array('ytg_Framework_Helper_Filter', 'filterList')),
            array('negative_keywords', 'match',
                'not' => TRUE,
                'pattern' => '~\-~isu',
                'message' => '{attribute} does not allow minus (-) character',
            ),
            array('negative_keywords', 'default'),


            array('post_type', 'in',
                'range' => ytg_Core::$app->postTypes->getContentTypes(),
            ),
            array('post_type', 'required'),
        );
    }

    //
    // Defaults
    //

    public function init()
    {
        $this->post_type = ytg_Core::config('option/default/postType');
    }

    //
    // Relations
    //

    public function getTemplate()
    {
        return ytg_Core::model('Model_Template')->findOne(array(
            'id' => $this->getTemplateId(),
        ));
    }

    //
    // Dynamic values
    //

    public function getTemplateId()
    {
        $result = $this->getAttribute('template_id');
        if (is_null($result)) {
            $result = 1;
        }

        return $result;
    }

    public function getChannelCategoryTable()
    {
        ytg_Core::load('Framework_Db_Schema');

        return ytg_Framework_Db_Schema::composeFullTableName('channel_category');
    }

    public function getCategories()
    {
        global $wpdb;

        if (is_null($this->_categories)) {
            if (!$this->isNewRecord) {
                $this->_categories = $wpdb->get_col(
                    "SELECT `category_id` 
                    FROM `{$this->getChannelCategoryTable()}` 
                    WHERE `channel_id` = {$this->id}");
            } else {
                $this->_categories = array();
            }
        }

        return $this->_categories;
    }

    public function setCategories($value)
    {
        if (is_null($this->_oldCategories)) {
            $this->_oldCategories = $this->getCategories();
        }

        $this->_categories = $value;

        return $this;
    }

    //
    // Lists
    //

    public function getTemplateList()
    {
        return ytg_Core::model('Model_Template')->getList();
    }

    public function getPostTypeList()
    {
        return ytg_Core::$app->postTypes->getList();
    }

    //
    // Validation
    //

    static public function extractChannelId($value)
    {
        if (preg_match('~^https?://(?:www\.)?youtube\.com/channel/([^/?]+)+~is',
            $value, $match)
        ) {
            $value = $match[1];
        }

        return $value;
    }

    //
    // Event handlers
    //

    public function beforeSave($insert)
    {
        if ($insert) {
            $this->last_check_timestamp = time();
        }

        if ('' == $this->name) {
            $this->name = $this->youtube_id;
        }

        return parent::beforeSave($insert);
    }

    public function afterSave($insert, $changedAttributes)
    {
        parent::afterSave($insert, $changedAttributes);

        if (!is_null($this->_oldCategories)) {
            global $wpdb;

            $tableName = $this->getChannelCategoryTable();

            if (!$insert) {
                $wpdb->delete($tableName,
                    array('channel_id' => $this->id));
            }

            foreach ((array) $this->_categories as $id) {
                $wpdb->insert($tableName, array(
                    'channel_id' => $this->id,
                    'category_id' => $id,
                ));
            }

            $this->_oldCategories = NULL;
        }
    }

    public function onCategoryDelete($id)
    {
        global $wpdb;

        $wpdb->delete($this->getChannelCategoryTable(),
            array('category_id' => $id));

        return $this;
    }
}
