<?php
ytg_Core::load('Framework_Service');
ytg_Core::load('Framework_Helper_Http');

class ytg_Component_Generator_Publisher extends ytg_Framework_Service
{
    /**
     * @var ytg_Component_Generator
     */
    public $generator;

    public $publicationDelay = 0;  //removed in 2.0
    public $publicationStartTime = 36000; //10:00
    public $publicationRange = 28800; //8 hours

    public $publicationDeltaRatio = 0.25;  // Random post offset +/- as a part of post step

    public $maxRetryCount = 3;

    protected $_publication;

    // Do not save these item fields as meta fields
    protected $_metaExclude = array(
        'content',
        'title',
        'tags',
    );

    public function init()
    {
        if (!$this->generator) {
            $this->generator = ytg_Core::$app->generator;
        }
    }

    public function publish(array &$item)
    {
        /**
         * @var ytg_Framework_Component_View $view
         */
        $view = ytg_Core::$app->view;

        $content = $view->render('generator/post', array(
            'content' => $view->escape($item['content']),
            'item' => $item,
        ));

        $post = array(
            'post_type' => $this->generator->postType,
            'post_status' => 'draft',
        );
        $post['post_content'] = $content;
        $post['post_title'] = $item['title'];

        $post['tags_input'] = implode(', ', $item['tags']);
        $post['post_category'] = $this->generator->postCategories;

        $scheduleTime = $this->_getTime();
        $postStatus = 'publish';
        if (!is_null($scheduleTime)) {
            $postStatus = 'future';
        } else if ('draft' == $this->generator->publicationMethod) {
            $postStatus = 'draft';
        }

        // Write post
        $postId = wp_insert_post($post);
        if (is_a($postId, 'WP_Error')) {
            throw new ytg_Component_Generator_Publisher_Exception($postId->get_error_message());
        } else if (!$postId) {
            global $wpdb;
            if ('' != $wpdb->last_error) {
                throw new ytg_Component_Generator_Publisher_Exception('Database error ' . $wpdb->last_error);
            }
            throw new ytg_Component_Generator_Publisher_Exception('Unknown error.');
        }

        // Set post format
        set_post_format($postId, 'video');

        // Save metadata
        $this->_saveMeta($postId, $item);

        if ('draft' != $postStatus) {
            // Update to real status
            $update = array(
                'ID' => $postId,
                'post_status' => $postStatus,
            );
            if (!is_null($scheduleTime)) {
                $update['post_date'] = get_date_from_gmt(gmdate('Y-m-d H:i:s',
                    $scheduleTime));
                $update['edit_date'] = $update['post_date']; // edit_date is required to force date change
            }
            wp_update_post($update);
        }

        $item['postId'] = $postId;

        return $postId;
    }

    protected function _saveMeta($postId, array &$item)
    {
        /**
         * @var ytg_Framework_Component_Meta $meta
         */
        $meta = ytg_Core::create('Framework_Component_Meta', array(
            'id' => $postId,
        ));

        $meta->set('version', ytg_Core::$app->version);

        $exclude = $this->_metaExclude;

        foreach ($item as $key => $val) {
            if (!in_array($key, $exclude)) {
                $meta->set($key, $val);
            }
        }
    }

    public function downloadFeaturedImage($url, array &$item)
    {
        /*
        $attachmentId = get_post_meta($postId, '_thumbnail_id', TRUE);
        if ('' != $attachmentId) {
            $attachment = get_post($attachmentId);
            if ($attachment && 'attachment' == $attachment->post_type) {
                $originalUrl = Asg_Application::getModel('PostMeta')->setPostId($attachmentId)
                    ->getValue('original_url');
                if ('' == $originalUrl || $url == $originalUrl) {
                    // Custom featured image or the same image
                    return;
                }
            }

            wp_delete_attachment($attachmentId, TRUE);
            delete_post_meta($postId, '_thumbnail_id');
        }
        */

        $filename = str_replace(array('%', '+', ' '), '-',
            urldecode(basename($url)));

        $uploadDir = wp_upload_dir();
        $filename = wp_unique_filename($uploadDir['path'], $filename);

        $savePath = $uploadDir['path'] . '/' . $filename;
        $saveUrl = $uploadDir['url'] . '/' . urlencode($filename);

        $fp = fopen($savePath, 'w+');
        if (!$fp) {
            throw new ytg_Component_Generator_Publisher_Exception("Could not create featured image file '{$savePath}'");
        }

        $curl = ytg_Framework_Helper_Http::initCurl($url, array(
            CURLOPT_RETURNTRANSFER => FALSE,
            CURLOPT_FILE => $fp,
            //CURLOPT_FOLLOWLOCATION => TRUE,
        ));

        for ($iRetry = 0; $iRetry <= $this->maxRetryCount; $iRetry++) {
            $result = curl_exec($curl);
            if (FALSE === $result) {
                if ($iRetry < $this->maxRetryCount) {
                    continue;
                }

                $error = curl_error($curl);

                curl_close($curl);
                fclose($fp);
                @unlink($savePath);
                throw new ytg_Component_Generator_Publisher_Exception($error);
            }

            $code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            if (200 != $code) {
                if ($iRetry < $this->maxRetryCount) {
                    continue;
                }

                curl_close($curl);
                fclose($fp);
                @unlink($savePath);
                throw new ytg_Component_Generator_Publisher_Exception("HTTP error #{$code}");
            }

            break;
        }

        fclose($fp);

        $result = wp_insert_attachment(array(
            'post_title' => $item['title'],
            'post_mime_type' => curl_getinfo($curl, CURLINFO_CONTENT_TYPE),
            'guid' => $saveUrl,
        ), $savePath, $item['postId']);

        curl_close($curl);

        // create multiple sizes of the uploaded image via WP controls
        if (!function_exists('wp_generate_attachment_metadata')) {
            require_once ABSPATH . 'wp-admin/includes/image.php';
        }
        wp_update_attachment_metadata($result,
            wp_generate_attachment_metadata($result, $savePath));

        // Save original URL
        update_post_meta(
            $result,
            ytg_Core::$app->prefix . '_originalUrl',
            $url
        );

        // Attach image reference
        update_post_meta($item['postId'], '_thumbnail_id', $result);

        return $result;
    }

    protected function _getTime()
    {
        if ('schedule' != $this->generator->publicationMethod) {
            return NULL;
        }

        if (!$this->_publication) {
            $start = time() + $this->publicationDelay;
            $day = floor($start/86400);
            $time = $start % 86400;
            //$step = $this->_scheduleSettings['avg'] > 1
            //    ? $this->_publicationRange / ($this->_scheduleSettings['avg'] - 1)
            //    : 0;

            $schedule = $this->getSchedule();

            $todaysPosts = $schedule['avg'];
            $endTime = $this->publicationStartTime + $this->publicationRange;
            $weekSchedule = $this->_weekSchedule();
            if (!(($todaysPosts > 1) && (
                    ($time>=$this->publicationStartTime && $time<$endTime)
                    || ($endTime>=86400 && $time<($endTime-86400))))
            ) {
                // Nothing to publish today. Move to tomorrow
                $this->_publication = array(
                    'day' => $day+1,
                    'todaysPosts' => $weekSchedule[0],
                    'postCounter' => 0,
                    'dayCounter' => 0,
                    'weekSchedule' => $weekSchedule,
                );
                return NULL;
            }

            // If posting overnight:
            if ($endTime>=86400 && $time<($endTime-86400)) {
                $time += 86400;
                $day--;
            }
            $step = $this->publicationRange / ($todaysPosts);
            $counter = floor(($time - $this->publicationStartTime)/$step);

            $this->_publication = array(
                'day' => $day,
                'todaysPosts' => $todaysPosts,
                'postCounter' => $counter,
                'dayCounter' => -1,
                'weekSchedule' => $weekSchedule,
            );

            return NULL;
        }

        $postCounter = $this->_publication['postCounter'];
        if ($postCounter >= $this->_publication['todaysPosts']) {
            while (true) {
                //Move to the next day
                $this->_publication['day']++;
                $dayCounter = ++$this->_publication['dayCounter'];
                if ($dayCounter >= 7) {
                    $dayCounter = $this->_publication['dayCounter'] = 0;
                    $this->_publication['weekSchedule'] = $this->_weekSchedule();
                }
                $todaysPosts = $this->_publication['weekSchedule'][$dayCounter];
                if ($todaysPosts > 0) {
                    $this->_publication['todaysPosts'] = $todaysPosts;
                    $postCounter = $this->_publication['postCounter'] = 0;
                    break;
                }
            }
        }

        $day = $this->_publication['day'];
        $todaysPosts = $this->_publication['todaysPosts'];

        if ($todaysPosts > 1) {
            // Set first post time
            if (0 == $postCounter || !isset($this->_publication['step'])) {
                $this->_publication['step'] = $this->publicationRange / $todaysPosts;
            }
            $step = $this->_publication['step'];
            $delta = (int)($step * $this->publicationDeltaRatio);
            $result = $step * ($postCounter + 0.5) + rand(-$delta, $delta);
        } else {
            // 1 post per day: any time whithin publication range
            $result = rand(0, $this->publicationRange);
        }
        $this->_publication['postCounter']++;

        $result += $this->publicationStartTime + (86400 * $day);
        if ($result < time()) {
            return NULL;
        }

        return $result;
    }

    protected function _weekSchedule()
    {
        $schedule = $this->getSchedule();

        $total = $schedule['weekly'];
        $min = $schedule['min'];
        $max = $schedule['max'];

        $lottery = array();
        for ($i = 0; $i < 7; $i++) {
            $lottery[$i] = array(
                'index' => $i,
                'count' => $min,
            );
        }
        $result = array();

        $count = $total - 7*$min;
        for ($i = 0; $i < $count; $i++) {
            $item = rand(0, count($lottery)-1);
            $lottery[$item]['count']++;
            if ($lottery[$item]['count'] >= $max) {
                $result[$lottery[$item]['index']] = $max;
                unset($lottery[$item]);
                // Reindex array
                $lottery = array_values($lottery);
            }
        }

        foreach ($lottery as $val) {
            $result[$val['index']] = $val['count'];
        }

        return $result;
    }

    public function getSchedule()
    {
        switch (ytg_Core::$app->generator->postsPerWeek) {
            case 3:
                return array(
                    'min' => 0,
                    'avg' => 0,
                    'max' => 1,
                    'weekly' => 3,
                );

            case 7:
                return array(
                    'min' => 0,
                    'avg' => 2,
                    'max' => 3,
                    'weekly' => 7,
                );

            case 15:
                return array(
                    'min' => 0,
                    'avg' => 2,
                    'max' => 4,
                    'weekly' => 15,
                );

            case 30:
                return array(
                    'min' => 2,
                    'avg' => 4,
                    'max' => 6,
                    'weekly' => 30,
                );

            case 100:
                return array(
                    'min' => 12,
                    'avg' => 14,
                    'max' => 16,
                    'weekly' => 100,
                );

            case 1:
            default:
                return array(
                    'min' => 0,
                    'avg' => 0,
                    'max' => 1,
                    'weekly' => 1,
                );
        }
    }

    public function addComments(&$item, WP_Post $post)
    {
        $this->generator->feedback("Adding comments... ");

        try {
            $result = ytg_Core::$app->client_Youtube->getVideoComments($item['videoId'], array(
                'textFormat' => 'plainText',
                'order' => 'relevance',
                'part' => 'snippet,replies',
            ));
        } catch (ytg_Component_Client_Youtube_Exception $e) {
            if ('commentsDisabled' == $e->reason) {
                $this->generator->feedback("comments are disabled for the video.\n");
                return;
            }

            throw new ytg_Component_Process_FatalException("Error getting video comments from YouTube: {$e->getMessage()}");
        }

        if (!$result['items']) {
            $this->generator->feedback("no comments found.\n");
            return;
        }

        $commentCount = 0;
        $replyCount = 0;
        $filteredCount = 0;
        $errorCount = 0;
        foreach ($result['items'] as $comment) {
            try {
                $parentId = $this->_addComment(
                    $comment['snippet']['topLevelComment']['snippet'],
                    $post);

                $commentCount++;
            } catch (ytg_Component_Generator_Publisher_BadWordsException $e) {
                $filteredCount++;
                continue;
            } catch (ytg_Component_Generator_Publisher_Exception $e) {
                $errorCount++;
                continue;
            }

            if (isset($comment['replies']['comments'])) {
                foreach ($comment['replies']['comments'] as $reply) {
                    try {
                        $this->_addComment(
                            $reply['snippet'],
                            $post,
                            $parentId);

                        $replyCount++;
                    } catch (ytg_Component_Generator_Publisher_BadWordsException $e) {
                        $filteredCount++;
                    } catch (ytg_Component_Generator_Publisher_Exception $e) {
                        $errorCount++;
                    }
                }
            }
        }

        if ($commentCount) {
            $this->generator->feedback("ok, added comments: {$commentCount}, replies: {$replyCount}, filtered out: {$filteredCount}, errors: {$errorCount}.\n");
        } else {
            $this->generator->feedback("no comment added, filtered out: {$filteredCount}, errors: {$errorCount}.\n");
        }
    }

    protected function _addComment(array $snippet, WP_Post $post, $parentId = NULL)
    {
        $text = $snippet['textDisplay'];

        if (ytg_Core::$app->badWordsFilter->hasBadWords($text)) {
            throw new ytg_Component_Generator_Publisher_BadWordsException('Bad words found');
        }

        $text = wp_check_invalid_utf8($text, TRUE);

        $result = wp_insert_comment(array(
            'comment_post_ID' => $post->ID,
            'comment_author' => $snippet['authorDisplayName'],
            //'comment_author_email' => 'admin@admin.com',
            //'comment_author_url' => 'http://',
            'comment_content' => $text,
            'comment_type' => '',
            'comment_parent' => $parentId,
            //'user_id' => 1,
            //'comment_author_IP' => '127.0.0.1',
            //'comment_agent' => 'Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.0.10) Gecko/2009042316 Firefox/3.0.10 (.NET CLR 3.5.30729)',
            'comment_date' => $this->_convertYoutubeDate($snippet['publishedAt']),
            'comment_date_gmt' => $this->_convertYoutubeDate($snippet['publishedAt'], TRUE),
            'comment_approved' => 1,
        ));

        if (!$result) {
            global $wpdb;
            if ('' != $wpdb->last_error) {
                throw new ytg_Component_Generator_Publisher_Exception('Database error while creating comment: ' . $wpdb->last_error);
            } else {
                throw new ytg_Component_Generator_Publisher_Exception('Unknown error while creating comment');
            }
        }

        $meta = ytg_Core::create('Framework_Component_Meta', array(
            'type' => 'comment',
            'id' => $result,
        ));
        $meta->set('info', $snippet);

        return $result;
    }

    protected function _convertYoutubeDate($date, $gmt=FALSE)
    {
        $result = strtotime($date);
        if (!$result) {
            return NULL;
        }

        if (!$gmt) {
            $result += get_option( 'gmt_offset' ) * HOUR_IN_SECONDS;
        }

        return gmdate('Y-m-d H:i:s', $result);
    }
}

class ytg_Component_Generator_Publisher_Exception extends Exception
{

}

class ytg_Component_Generator_Publisher_BadWordsException
    extends ytg_Component_Generator_Exception
{

}